/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/frechet/cdf' );
import Frechet = require( './../../../../../base/dists/frechet/ctor' );
import entropy = require( './../../../../../base/dists/frechet/entropy' );
import kurtosis = require( './../../../../../base/dists/frechet/kurtosis' );
import logcdf = require( './../../../../../base/dists/frechet/logcdf' );
import logpdf = require( './../../../../../base/dists/frechet/logpdf' );
import mean = require( './../../../../../base/dists/frechet/mean' );
import median = require( './../../../../../base/dists/frechet/median' );
import mode = require( './../../../../../base/dists/frechet/mode' );
import pdf = require( './../../../../../base/dists/frechet/pdf' );
import quantile = require( './../../../../../base/dists/frechet/quantile' );
import skewness = require( './../../../../../base/dists/frechet/skewness' );
import stdev = require( './../../../../../base/dists/frechet/stdev' );
import variance = require( './../../../../../base/dists/frechet/variance' );

/**
* Interface describing the `frechet` namespace.
*/
interface Namespace {
	/**
	* Fréchet distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param alpha - shape parameter
	* @param s - scale parameter
	* @param m - location parameter
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 10.0, 2.0, 3.0, 5.0 );
	* // returns ~0.698
	*
	* y = ns.cdf( 0.0, 2.0, 3.0, 2.0 );
	* // returns 0.0
	*
	* var mycdf = ns.cdf.factory( 3.0, 3.0, 5.0 );
	* y = mycdf( 10.0 );
	* // returns ~0.806
	*
	* y = mycdf( 7.0 );
	* // returns ~0.034
	*/
	cdf: typeof cdf;

	/**
	* Frechet Distribution.
	*/
	Frechet: typeof Frechet;

	/**
	* Returns the differential entropy for a Fréchet distribution with shape `alpha`, scale `s`, and location `m`.
	*
	* ## Notes
	*
	* -   If provided `alpha <= 0` or `s <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param s - scale parameter
	* @param m - location parameter
	* @returns entropy
	*
	* @example
	* var y = ns.entropy( 1.0, 1.0, 0.0 );
	* // returns ~2.154
	*
	* @example
	* var y = ns.entropy( 5.0, 2.0, 0.0 );
	* // returns ~0.776
	*
	* @example
	* var y = ns.entropy( NaN, 1.0, 0.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.entropy( 1.0, NaN, 0.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.entropy( 1.0, 1.0, NaN );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis for a Fréchet distribution with shape `alpha`, scale `s`, and location `m`.
	*
	* ## Notes
	*
	* -   If provided `0 < alpha <= 4` and `s > 0`, the function returns positive infinity.
	* -   If provided `alpha <= 0` or `s <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param s - scale parameter
	* @param m - location parameter
	* @returns excess kurtosis
	*
	* @example
	* var y = ns.kurtosis( 5.0, 2.0, 0.0 );
	* // returns ~45.092
	*
	* @example
	* var y = ns.kurtosis( 5.0, 2.0, -5.0 );
	* // returns ~45.092
	*
	* @example
	* var y = ns.kurtosis( 3.8, 1.0, 0.0 );
	* // returns Infinity
	*
	* @example
	* var y = ns.kurtosis( NaN, 1.0, 0.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.kurtosis( 1.0, NaN, 0.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.kurtosis( 1.0, 1.0, NaN );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Fréchet distribution logarithm of cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param alpha - shape parameter
	* @param s - scale parameter
	* @param m - location parameter
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 10.0, 2.0, 3.0, 5.0 );
	* // returns ~-0.36
	*
	* y = ns.logcdf( 3.8, 2.0, 3.0, 2.0 );
	* // returns ~-2.778
	*
	* var mylogcdf = ns.logcdf.factory( 3.0, 3.0, 5.0 );
	* y = mylogcdf( 10.0 );
	* // returns ~-0.216
	*
	* y = mylogcdf( 7.0 );
	* // returns ~-3.381
	*/
	logcdf: typeof logcdf;

	/**
	* Fréchet distribution natural logarithm of the probability density function (logPDF).
	*
	* @param x - input value
	* @param alpha - shape parameter
	* @param s - scale parameter
	* @param m - location parameter
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 10.0, 2.0, 3.0, 5.0 );
	* // returns ~-2.298
	*
	* y = ns.logpdf( 0.0, 2.0, 3.0, 2.0 );
	* // returns -Infinity
	*
	* var mylogpdf = ns.logpdf.factory( 3.0, 3.0, 5.0 );
	* y = mylogpdf( 10.0 );
	* // returns ~-2.259
	*
	* y = mylogpdf( 7.0 );
	* // returns ~-1.753
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the expected value for a Fréchet distribution with shape `alpha`, scale `s`, and location `m`.
	*
	* ## Notes
	*
	* -   If provided `0 < alpha <= 1` and `s > 0`, the function returns positive infinity.
	* -   If provided `alpha <= 0` or `s <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param s - scale parameter
	* @param m - location parameter
	* @returns mean
	*
	* @example
	* var y = ns.mean( 5.0, 2.0, 0.0 );
	* // returns ~2.328
	*
	* @example
	* var y = ns.mean( 5.0, 2.0, -4.0 );
	* // returns ~-1.672
	*
	* @example
	* var y = ns.mean( 1.0, 1.0, 0.0 );
	* // returns Infinity
	*
	* @example
	* var y = ns.mean( NaN, 1.0, 0.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.mean( 1.0, NaN, 0.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.mean( 1.0, 1.0, NaN );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median for a Fréchet distribution with shape `alpha`, scale `s`, and location `m`.
	*
	* ## Notes
	*
	* -   If provided `alpha <= 0` or `s <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param s - scale parameter
	* @param m - location parameter
	* @returns median
	*
	* @example
	* var y = ns.median( 5.0, 2.0, 0.0 );
	* // returns ~2.152
	*
	* @example
	* var y = ns.median( 5.0, 2.0, -5.0 );
	* // returns ~-2.848
	*
	* @example
	* var y = ns.median( 1.0, 1.0, 0.0 );
	* // returns ~1.443
	*
	* @example
	* var y = ns.median( NaN, 1.0, 0.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.median( 1.0, NaN, 0.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.median( 1.0, 1.0, NaN );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Returns the mode for a Fréchet distribution with shape `alpha`, scale `s`, and location `m`.
	*
	* ## Notes
	*
	* -   If provided `alpha <= 0` or `s <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param s - scale parameter
	* @param m - location parameter
	* @returns mode
	*
	* @example
	* var y = ns.mode( 5.0, 2.0, 0.0 );
	* // returns ~1.928
	*
	* @example
	* var y = ns.mode( 5.0, 2.0, -5.0 );
	* // returns ~-3.072
	*
	* @example
	* var y = ns.mode( 1.0, 1.0, 0.0 );
	* // returns ~0.5
	*
	* @example
	* var y = ns.mode( NaN, 1.0, 0.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.mode( 1.0, NaN, 0.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.mode( 1.0, 1.0, NaN );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Fréchet distribution probability density function (PDF).
	*
	* @param x - input value
	* @param alpha - shape parameter
	* @param s - scale parameter
	* @param m - location parameter
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 10.0, 2.0, 3.0, 5.0 );
	* // returns ~0.698
	*
	* y = ns.pdf( 0.0, 2.0, 3.0, 2.0 );
	* // returns 0.0
	*
	* var mypdf = ns.pdf.factory( 3.0, 3.0, 5.0 );
	* y = mypdf( 10.0 );
	* // returns ~0.806
	*
	* y = mypdf( 7.0 );
	* // returns ~0.034
	*/
	pdf: typeof pdf;

	/**
	* Fréchet distribution quantile function.
	*
	* @param p - input value
	* @param alpha - shape parameter
	* @param s - scale parameter
	* @param m - location parameter
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.3, 2.0, 3.0, 5.0 );
	* // returns ~7.734
	*
	* y = ns.quantile( 0.8, 2.0, 3.0, 2.0 );
	* // returns ~8.351
	*
	* var myquantile = ns.quantile.factory( 3.0, 3.0, 5.0 );
	* y = myquantile( 0.1 );
	* // returns ~7.272
	*
	* y = myquantile( 0.8 );
	* // returns ~9.946
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness for a Fréchet distribution with shape `alpha`, scale `s`, and location `m`.
	*
	* ## Notes
	*
	* -   If provided `0 < alpha <= 3` and `s > 0`, the function returns positive infinity.
	* -   If provided `alpha <= 0` or `s <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param s - scale parameter
	* @param m - location parameter
	* @returns skewness
	*
	* @example
	* var y = ns.skewness( 5.0, 2.0, 0.0 );
	* // returns ~3.535
	*
	* @example
	* var y = ns.skewness( 5.0, 2.0, -5.0 );
	* // returns ~3.535
	*
	* @example
	* var y = ns.skewness( 1.0, 1.0, 0.0 );
	* // returns Infinity
	*
	* @example
	* var y = ns.skewness( NaN, 1.0, 0.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.skewness( 1.0, NaN, 0.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.skewness( 1.0, 1.0, NaN );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation for a Fréchet distribution with shape `alpha`, scale `s`, and location `m`.
	*
	* ## Notes
	*
	* -   If provided `0 < alpha <= 2` and `s > 0`, the function returns positive infinity.
	* -   If provided `alpha <= 0` or `s <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param s - scale parameter
	* @param m - location parameter
	* @returns standard deviation
	*
	* @example
	* var y = ns.stdev( 5.0, 2.0, 0.0 );
	* // returns ~0.731
	*
	* @example
	* var y = ns.stdev( 5.0, 2.0, -5.0 );
	* // returns ~0.731
	*
	* @example
	* var y = ns.stdev( 1.0, 1.0, 0.0 );
	* // returns Infinity
	*
	* @example
	* var y = ns.stdev( NaN, 1.0, 0.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.stdev( 1.0, NaN, 0.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.stdev( 1.0, 1.0, NaN );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance for a Fréchet distribution with shape `alpha`, scale `s`, and location `m`.
	* ## Notes
	*
	* -   If provided `0 < alpha <= 2` and `s > 0`, the function returns positive infinity.
	* -   If provided `alpha <= 0` or `s <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param s - scale parameter
	* @param m - location parameter
	* @returns variance
	*
	* @example
	* var y = ns.variance( 5.0, 2.0, 0.0 );
	* // returns ~0.535
	*
	* @example
	* var y = ns.variance( 5.0, 2.0, -5.0 );
	* // returns ~0.535
	*
	* @example
	* var y = ns.variance( 1.0, 1.0, 0.0 );
	* // returns Infinity
	*
	* @example
	* var y = ns.variance( NaN, 1.0, 0.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.variance( 1.0, NaN, 0.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.variance( 1.0, 1.0, NaN );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Fréchet distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
