// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

/*****************************************************************************/
#ifndef HARDWARE_INTRINSIC
#error Define HARDWARE_INTRINSIC before including this file
#endif
/*****************************************************************************/

// clang-format off

#ifdef FEATURE_HW_INTRINSICS
/* Note
    1) Each hardware intrinsic has a unique Intrinsic ID with type of `enum NamedIntrinsic`
    2) All the overloads of an intrinsic in an ISA class share one Intrinsic ID
    3) The intrinsic that generates instructions with a fixed imm8 operand has a `ival` field with "not -1" value, e.g., Sse.CompareEqual(v1,v2) -> cmpps xmm0, xmm1, 0
    4) SIMD intrinsics have a non-zero `SIMD size` field based-on that operate over `Vector128<T>`(16) or `Vector256<T>`(32)
    5) Scalar intrinsics that operate over general purpose registers (e.g., Sse41.Crc32) have `SIMD size` with 0
    6) Each intrinsic has a `NumArg` for number of parameters, and some intrinsics that are overloaded on multiple parameter numbers have this field with -1
    7) Each intrinsic has 10 `instructions` fields that list the instructions should be generated based-on the base type
    8) Each intrinsic has one category with type of `enum HWIntrinsicCategory`, please see the definition of HWIntrinsicCategory for details
    9) Each intrinsic has one or more flags with type of `enum HWIntrinsicFlag`
*/

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                              {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  Vector128 Intrinsics
#define FIRST_NI_Vector128          NI_Vector128_Abs
HARDWARE_INTRINSIC(Vector128,       Abs,                                        16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       AndNot,                                     16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       As,                                         16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       AsByte,                                     16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       AsDouble,                                   16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       AsInt16,                                    16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       AsInt32,                                    16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       AsInt64,                                    16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       AsNInt,                                     16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       AsNUInt,                                    16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       AsSByte,                                    16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       AsSingle,                                   16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       AsUInt16,                                   16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       AsUInt32,                                   16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       AsUInt64,                                   16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       AsVector,                                   16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       AsVector128,                                -1,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       AsVector128Unsafe,                          -1,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movups,             INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_NoRMWSemantics|HW_Flag_NoContainment)
HARDWARE_INTRINSIC(Vector128,       AsVector2,                                  16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movsd_simd,         INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(Vector128,       AsVector3,                                  16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movups,             INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(Vector128,       AsVector4,                                  16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       Ceiling,                                    16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       ConditionalSelect,                          16,             3,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen)
HARDWARE_INTRINSIC(Vector128,       ConvertToDouble,                            16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       ConvertToInt32,                             16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       ConvertToInt32Native,                       16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       ConvertToInt64,                             16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       ConvertToInt64Native,                       16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       ConvertToSingle,                            16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       ConvertToUInt32,                            16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       ConvertToUInt32Native,                      16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       ConvertToUInt64,                            16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       ConvertToUInt64Native,                      16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       Create,                                     16,            -1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen)
HARDWARE_INTRINSIC(Vector128,       CreateScalar,                               16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen)
HARDWARE_INTRINSIC(Vector128,       CreateScalarUnsafe,                         16,             1,      {INS_movd,              INS_movd,               INS_movd,               INS_movd,               INS_movd,               INS_movd,               INS_movd,               INS_movd,               INS_movss,              INS_movsd_simd},        HW_Category_SIMDScalar,             HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(Vector128,       CreateSequence,                             16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       Dot,                                        16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       Equals,                                     16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       EqualsAny,                                  16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       ExtractMostSignificantBits,                 16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       Floor,                                      16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       FusedMultiplyAdd,                           16,             3,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       GetElement,                                 16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_extractps,          INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       GreaterThan,                                16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       GreaterThanAll,                             16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       GreaterThanAny,                             16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       GreaterThanOrEqual,                         16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       GreaterThanOrEqualAll,                      16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       GreaterThanOrEqualAny,                      16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       IsNaN,                                      16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       IsNegative,                                 16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       IsPositive,                                 16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       IsPositiveInfinity,                         16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       IsZero,                                     16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       LessThan,                                   16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       LessThanAll,                                16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       LessThanAny,                                16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       LessThanOrEqual,                            16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       LessThanOrEqualAll,                         16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       LessThanOrEqualAny,                         16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       LoadAligned,                                16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       LoadAlignedNonTemporal,                     16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       LoadUnsafe,                                 16,            -1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       Max,                                        16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       MaxNative,                                  16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       Min,                                        16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       MinNative,                                  16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       MultiplyAddEstimate,                        16,             3,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       Narrow,                                     16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       Round,                                      16,            -1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       Shuffle,                                    16,            -1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(Vector128,       Sqrt,                                       16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       StoreAligned,                               16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       StoreAlignedNonTemporal,                    16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       StoreUnsafe,                                16,            -1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       Sum,                                        16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       ToScalar,                                   16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movss,              INS_movsd_simd},        HW_Category_SIMDScalar,             HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(Vector128,       ToVector256,                                16,             1,      {INS_movdqu,            INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movups,             INS_movupd},            HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(Vector128,       ToVector256Unsafe,                          16,             1,      {INS_movdqu,            INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movups,             INS_movupd},            HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(Vector128,       ToVector512,                                16,             1,      {INS_movdqu,            INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_vmovdqu64,          INS_vmovdqu64,          INS_movups,             INS_movupd},            HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(Vector128,       Truncate,                                   16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       WidenLower,                                 16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       WidenUpper,                                 16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       WithElement,                                16,             3,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector128,       get_AllBitsSet,                             16,             0,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       get_Indices,                                16,             0,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       get_One,                                    16,             0,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       get_Zero,                                   16,             0,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       op_Addition,                                16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       op_BitwiseAnd,                              16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       op_BitwiseOr,                               16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       op_Division,                                16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       op_Equality,                                16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen|HW_Flag_Commutative|HW_Flag_CanBenefitFromConstantProp)
HARDWARE_INTRINSIC(Vector128,       op_ExclusiveOr,                             16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       op_Inequality,                              16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen|HW_Flag_Commutative|HW_Flag_CanBenefitFromConstantProp)
HARDWARE_INTRINSIC(Vector128,       op_LeftShift,                               16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       op_Multiply,                                16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       op_OnesComplement,                          16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       op_RightShift,                              16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       op_Subtraction,                             16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       op_UnaryNegation,                           16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       op_UnaryPlus,                               16,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector128,       op_UnsignedRightShift,                      16,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
#define LAST_NI_Vector128           NI_Vector128_op_UnsignedRightShift

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  Vector256 Intrinsics
#define FIRST_NI_Vector256          NI_Vector256_Abs
HARDWARE_INTRINSIC(Vector256,       Abs,                                        32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       AndNot,                                     32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       As,                                         32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       AsByte,                                     32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       AsDouble,                                   32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       AsInt16,                                    32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       AsInt32,                                    32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       AsInt64,                                    32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       AsNInt,                                     32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       AsNUInt,                                    32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       AsSByte,                                    32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       AsSingle,                                   32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       AsUInt16,                                   32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       AsUInt32,                                   32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       AsUInt64,                                   32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       AsVector,                                   32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       AsVector256,                                32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       Ceiling,                                    32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       ConditionalSelect,                          32,             3,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       ConvertToDouble,                            32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       ConvertToInt32,                             32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       ConvertToInt32Native,                       32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       ConvertToInt64,                             32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       ConvertToInt64Native,                       32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       ConvertToSingle,                            32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       ConvertToUInt32,                            32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       ConvertToUInt32Native,                      32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       ConvertToUInt64,                            32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       ConvertToUInt64Native,                      32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       Create,                                     32,            -1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       CreateScalar,                               32,            -1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       CreateScalarUnsafe,                         32,             1,      {INS_movd,              INS_movd,               INS_movd,               INS_movd,               INS_movd,               INS_movd,               INS_movd,               INS_movd,               INS_movss,              INS_movsd_simd},        HW_Category_SIMDScalar,             HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       CreateSequence,                             32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       Dot,                                        32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       Equals,                                     32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       EqualsAny,                                  32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       ExtractMostSignificantBits,                 32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       Floor,                                      32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       FusedMultiplyAdd,                           32,             3,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       GetElement,                                 32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       GetLower,                                   32,             1,      {INS_movdqu,            INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movups,             INS_movupd},            HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       GetUpper,                                   32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       GreaterThan,                                32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       GreaterThanAll,                             32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       GreaterThanAny,                             32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       GreaterThanOrEqual,                         32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       GreaterThanOrEqualAll,                      32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       GreaterThanOrEqualAny,                      32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       IsNaN,                                      32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       IsNegative,                                 32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       IsPositive,                                 32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       IsPositiveInfinity,                         32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       IsZero,                                     32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       LessThan,                                   32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       LessThanAll,                                32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       LessThanAny,                                32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       LessThanOrEqual,                            32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       LessThanOrEqualAll,                         32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       LessThanOrEqualAny,                         32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       LoadAligned,                                32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       LoadAlignedNonTemporal,                     32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       LoadUnsafe,                                 32,            -1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       Max,                                        32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       MaxNative,                                  32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       Min,                                        32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       MinNative,                                  32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       MultiplyAddEstimate,                        32,             3,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       Narrow,                                     32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       Round,                                      32,            -1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       Shuffle,                                    32,            -1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(Vector256,       Sqrt,                                       32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       StoreAligned,                               32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       StoreAlignedNonTemporal,                    32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       StoreUnsafe,                                32,            -1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       Sum,                                        32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       ToScalar,                                   32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movss,              INS_movsd_simd},        HW_Category_SIMDScalar,             HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       ToVector512,                                32,             1,      {INS_movdqu,            INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_vmovdqu64,          INS_vmovdqu64,          INS_movups,             INS_movupd},            HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       ToVector512Unsafe,                          32,             1,      {INS_movdqu,            INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_vmovdqu64,          INS_vmovdqu64,          INS_movups,             INS_movupd},            HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       Truncate,                                   32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       WidenLower,                                 32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       WidenUpper,                                 32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector256,       WithElement,                                32,             3,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       WithLower,                                  32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       WithUpper,                                  32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       get_AllBitsSet,                             32,             0,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       get_Indices,                                32,             0,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       get_One,                                    32,             0,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       get_Zero,                                   32,             0,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       op_Addition,                                32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       op_BitwiseAnd,                              32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       op_BitwiseOr,                               32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       op_Division,                                32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       op_Equality,                                32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen|HW_Flag_Commutative|HW_Flag_CanBenefitFromConstantProp)
HARDWARE_INTRINSIC(Vector256,       op_ExclusiveOr,                             32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       op_Inequality,                              32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen|HW_Flag_Commutative|HW_Flag_CanBenefitFromConstantProp)
HARDWARE_INTRINSIC(Vector256,       op_LeftShift,                               32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       op_Multiply,                                32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       op_OnesComplement,                          32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       op_RightShift,                              32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       op_Subtraction,                             32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       op_UnaryNegation,                           32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector256,       op_UnaryPlus,                               32,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_AvxOnlyCompatible)
HARDWARE_INTRINSIC(Vector256,       op_UnsignedRightShift,                      32,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
#define LAST_NI_Vector256           NI_Vector256_op_UnsignedRightShift

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  Vector512 Intrinsics
#define FIRST_NI_Vector512          NI_Vector512_Abs
HARDWARE_INTRINSIC(Vector512,       Abs,                                        64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       AndNot,                                     64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       As,                                         64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       AsByte,                                     64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       AsDouble,                                   64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       AsInt16,                                    64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       AsInt32,                                    64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       AsInt64,                                    64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       AsNInt,                                     64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       AsNUInt,                                    64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       AsSByte,                                    64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       AsSingle,                                   64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       AsUInt16,                                   64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       AsUInt32,                                   64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       AsUInt64,                                   64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       AsVector,                                   64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       AsVector512,                                64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       Ceiling,                                    64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       ConditionalSelect,                          64,             3,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen)
HARDWARE_INTRINSIC(Vector512,       ConvertToDouble,                            64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       ConvertToInt32,                             64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       ConvertToInt32Native,                       64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       ConvertToInt64,                             64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       ConvertToInt64Native,                       64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       ConvertToSingle,                            64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       ConvertToUInt32,                            64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       ConvertToUInt32Native,                      64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       ConvertToUInt64,                            64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       ConvertToUInt64Native,                      64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       Create,                                     64,            -1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen)
HARDWARE_INTRINSIC(Vector512,       CreateScalar,                               64,            -1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen)
HARDWARE_INTRINSIC(Vector512,       CreateScalarUnsafe,                         64,             1,      {INS_movd,              INS_movd,               INS_movd,               INS_movd,               INS_movd,               INS_movd,               INS_movd,               INS_movd,               INS_movss,              INS_movsd_simd},        HW_Category_SIMDScalar,             HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(Vector512,       CreateSequence,                             64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       Equals,                                     64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       EqualsAny,                                  64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       ExtractMostSignificantBits,                 64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       Floor,                                      64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       FusedMultiplyAdd,                           64,             3,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       GetElement,                                 64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       GetLower,                                   64,             1,      {INS_movdqu,            INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_vmovdqu64,          INS_vmovdqu64,          INS_movups,             INS_movupd},            HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(Vector512,       GetLower128,                                64,             1,      {INS_movdqu,            INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_movdqu,             INS_vmovdqu64,          INS_vmovdqu64,          INS_movups,             INS_movupd},            HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       GetUpper,                                   64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen)
HARDWARE_INTRINSIC(Vector512,       GreaterThan,                                64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       GreaterThanAll,                             64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       GreaterThanAny,                             64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       GreaterThanOrEqual,                         64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       GreaterThanOrEqualAll,                      64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       GreaterThanOrEqualAny,                      64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       IsNaN,                                      64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       IsNegative,                                 64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       IsPositive,                                 64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       IsPositiveInfinity,                         64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       IsZero,                                     64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       LessThan,                                   64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       LessThanAll,                                64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       LessThanAny,                                64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       LessThanOrEqual,                            64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       LessThanOrEqualAll,                         64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       LessThanOrEqualAny,                         64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       LoadAligned,                                64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       LoadAlignedNonTemporal,                     64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       LoadUnsafe,                                 64,            -1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       Max,                                        64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       MaxNative,                                  64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       Min,                                        64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       MinNative,                                  64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       MultiplyAddEstimate,                        64,             3,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       Narrow,                                     64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       Round,                                      64,            -1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       Shuffle,                                    64,            -1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(Vector512,       Sqrt,                                       64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       StoreAligned,                               64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       StoreAlignedNonTemporal,                    64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       StoreUnsafe,                                64,            -1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       Sum,                                        64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       ToScalar,                                   64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movss,              INS_movsd_simd},        HW_Category_SIMDScalar,             HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       Truncate,                                   64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       WidenLower,                                 64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       WidenUpper,                                 64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       WithElement,                                64,             3,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(Vector512,       WithLower,                                  64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen)
HARDWARE_INTRINSIC(Vector512,       WithUpper,                                  64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen)
HARDWARE_INTRINSIC(Vector512,       get_AllBitsSet,                             64,             0,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       get_Indices,                                64,             0,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       get_One,                                    64,             0,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       get_Zero,                                   64,             0,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       op_Addition,                                64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       op_BitwiseAnd,                              64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       op_BitwiseOr,                               64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       op_Division,                                64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       op_Equality,                                64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen|HW_Flag_Commutative|HW_Flag_CanBenefitFromConstantProp)
HARDWARE_INTRINSIC(Vector512,       op_ExclusiveOr,                             64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       op_Inequality,                              64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_SpecialImport|HW_Flag_NoCodeGen|HW_Flag_Commutative|HW_Flag_CanBenefitFromConstantProp)
HARDWARE_INTRINSIC(Vector512,       op_LeftShift,                               64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       op_Multiply,                                64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       op_OnesComplement,                          64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       op_RightShift,                              64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       op_Subtraction,                             64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       op_UnaryNegation,                           64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       op_UnaryPlus,                               64,             1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(Vector512,       op_UnsignedRightShift,                      64,             2,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
#define LAST_NI_Vector512           NI_Vector512_op_UnsignedRightShift

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  X86Base Intrinsics
#define FIRST_NI_X86Base            NI_X86Base_BitScanForward
HARDWARE_INTRINSIC(X86Base,         BitScanForward,                             0,              1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_bsf,                INS_bsf,                INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(X86Base,         BitScanReverse,                             0,              1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_bsr,                INS_bsr,                INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(X86Base,         DivRem,                                     0,              3,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_idiv,               INS_div,                INS_idiv,               INS_div,                INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_BaseTypeFromSecondArg|HW_Flag_MultiReg|HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen|HW_Flag_RmwIntrinsic)
HARDWARE_INTRINSIC(X86Base,         Pause,                                      0,              0,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment|HW_Flag_NoRMWSemantics|HW_Flag_SpecialSideEffect_Other)
#define LAST_NI_X86Base             NI_X86Base_Pause

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  X86Base 64-bit-only Intrinsics
#define FIRST_NI_X86Base_X64        NI_X86Base_X64_BitScanForward
HARDWARE_INTRINSIC(X86Base_X64,     BitScanForward,                             0,              1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_bsf,                INS_bsf,                INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(X86Base_X64,     BitScanReverse,                             0,              1,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_bsr,                INS_bsr,                INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(X86Base_X64,     DivRem,                                     0,              3,      {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_idiv,               INS_div,                INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_BaseTypeFromSecondArg|HW_Flag_MultiReg|HW_Flag_SpecialImport|HW_Flag_SpecialCodeGen|HW_Flag_RmwIntrinsic)
#define LAST_NI_X86Base_X64         NI_X86Base_X64_DivRem

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  SSE Intrinsics
#define FIRST_NI_SSE                NI_SSE_Add
HARDWARE_INTRINSIC(SSE,             Add,                                        16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_addps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE,             AddScalar,                                  16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_addss,              INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE,             And,                                        16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_andps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE,             AndNot,                                     16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_andnps,             INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE,             CompareEqual,                               16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareGreaterThan,                         16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareGreaterThanOrEqual,                  16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareLessThan,                            16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareLessThanOrEqual,                     16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareNotEqual,                            16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareNotGreaterThan,                      16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareNotGreaterThanOrEqual,               16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareNotLessThan,                         16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareNotLessThanOrEqual,                  16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareOrdered,                             16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarEqual,                         16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpss,              INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarGreaterThan,                   16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpss,              INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_SpecialImport|HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarGreaterThanOrEqual,            16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpss,              INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_SpecialImport|HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarLessThan,                      16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpss,              INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarLessThanOrEqual,               16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpss,              INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarNotEqual,                      16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpss,              INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarNotGreaterThan,                16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpss,              INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_SpecialImport|HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarNotGreaterThanOrEqual,         16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpss,              INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_SpecialImport|HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarNotLessThan,                   16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpss,              INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarNotLessThanOrEqual,            16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpss,              INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarOrdered,                       16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpss,              INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarOrderedEqual,                  16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_comiss,             INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_Commutative|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarOrderedGreaterThan,            16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_comiss,             INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarOrderedGreaterThanOrEqual,     16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_comiss,             INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarOrderedLessThan,               16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_comiss,             INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarOrderedLessThanOrEqual,        16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_comiss,             INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarOrderedNotEqual,               16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_comiss,             INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_Commutative|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarUnordered,                     16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpss,              INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarUnorderedEqual,                16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_ucomiss,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_Commutative|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarUnorderedGreaterThan,          16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_ucomiss,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarUnorderedGreaterThanOrEqual,   16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_ucomiss,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarUnorderedLessThan,             16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_ucomiss,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarUnorderedLessThanOrEqual,      16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_ucomiss,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE,             CompareScalarUnorderedNotEqual,             16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_ucomiss,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_Commutative|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE,             CompareUnordered,                           16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             ConvertScalarToVector128Single,             16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtsi2ss32,         INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromSecondArg|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE,             ConvertToInt32,                             16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtss2si,           INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE,             ConvertToInt32WithTruncation,               16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvttss2si32,        INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE,             Divide,                                     16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_divps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE,             DivideScalar,                               16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_divss,              INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE,             LoadAlignedVector128,                       16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movaps,             INS_invalid},           HW_Category_MemoryLoad,             HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE,             LoadHigh,                                   16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movhps,             INS_invalid},           HW_Category_MemoryLoad,             HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE,             LoadLow,                                    16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movlps,             INS_invalid},           HW_Category_MemoryLoad,             HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE,             LoadScalarVector128,                        16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movss,              INS_invalid},           HW_Category_MemoryLoad,             HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE,             LoadVector128,                              16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(SSE,             Max,                                        16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_maxps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_MaybeCommutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE,             MaxScalar,                                  16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_maxss,              INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE,             Min,                                        16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_minps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_MaybeCommutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE,             MinScalar,                                  16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_minss,              INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE,             MoveHighToLow,                              16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movhlps,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoContainment)
HARDWARE_INTRINSIC(SSE,             MoveLowToHigh,                              16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movlhps,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoContainment)
HARDWARE_INTRINSIC(SSE,             MoveMask,                                   16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movmskps,           INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoContainment|HW_Flag_NoRMWSemantics|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             MoveScalar,                                 16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movss,              INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_NoContainment)
HARDWARE_INTRINSIC(SSE,             Multiply,                                   16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_mulps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE,             MultiplyScalar,                             16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_mulss,              INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE,             Or,                                         16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_orps,               INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE,             Prefetch0,                                   0,              1,     {INS_invalid,           INS_prefetcht0,         INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment|HW_Flag_NoRMWSemantics|HW_Flag_SpecialSideEffect_Other)
HARDWARE_INTRINSIC(SSE,             Prefetch1,                                   0,              1,     {INS_invalid,           INS_prefetcht1,         INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment|HW_Flag_NoRMWSemantics|HW_Flag_SpecialSideEffect_Other)
HARDWARE_INTRINSIC(SSE,             Prefetch2,                                   0,              1,     {INS_invalid,           INS_prefetcht2,         INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment|HW_Flag_NoRMWSemantics|HW_Flag_SpecialSideEffect_Other)
HARDWARE_INTRINSIC(SSE,             PrefetchNonTemporal,                         0,              1,     {INS_invalid,           INS_prefetchnta,        INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment|HW_Flag_NoRMWSemantics|HW_Flag_SpecialSideEffect_Other)
HARDWARE_INTRINSIC(SSE,             Reciprocal,                                 16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_rcpps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoRMWSemantics|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             ReciprocalScalar,                           16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_rcpss,              INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             ReciprocalSqrt,                             16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_rsqrtps,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoRMWSemantics|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             ReciprocalSqrtScalar,                       16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_rsqrtss,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE,             Shuffle,                                    16,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_shufps,             INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE,             Sqrt,                                       16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_sqrtps,             INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE,             SqrtScalar,                                 16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_sqrtss,             INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE,             Store,                                      16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(SSE,             StoreAligned,                               16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movaps,             INS_invalid},           HW_Category_MemoryStore,            HW_Flag_NoRMWSemantics|HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(SSE,             StoreAlignedNonTemporal,                    16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movntps,            INS_invalid},           HW_Category_MemoryStore,            HW_Flag_NoRMWSemantics|HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(SSE,             StoreFence,                                  0,              0,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment|HW_Flag_NoRMWSemantics|HW_Flag_SpecialSideEffect_Barrier)
HARDWARE_INTRINSIC(SSE,             StoreHigh,                                  16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movhps,             INS_invalid},           HW_Category_MemoryStore,            HW_Flag_NoRMWSemantics|HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(SSE,             StoreLow,                                   16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movlps,             INS_invalid},           HW_Category_MemoryStore,            HW_Flag_NoRMWSemantics|HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(SSE,             StoreScalar,                                16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movss,              INS_invalid},           HW_Category_MemoryStore,            HW_Flag_NoRMWSemantics|HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(SSE,             Subtract,                                   16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_subps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE,             SubtractScalar,                             16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_subss,              INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE,             UnpackHigh,                                 16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_unpckhps,           INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE,             UnpackLow,                                  16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_unpcklps,           INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE,             Xor,                                        16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_xorps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_CanBenefitFromConstantProp)
#define LAST_NI_SSE                 NI_SSE_Xor

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  SSE 64-bit-only Intrinsics
#define FIRST_NI_SSE_X64            NI_SSE_X64_ConvertScalarToVector128Single
HARDWARE_INTRINSIC(SSE_X64,         ConvertScalarToVector128Single,             16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtsi2ss64,         INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromSecondArg|HW_Flag_CopyUpperBits|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(SSE_X64,         ConvertToInt64,                             16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtss2si,           INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(SSE_X64,         ConvertToInt64WithTruncation,               16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvttss2si64,        INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics|HW_Flag_SpecialCodeGen)
#define LAST_NI_SSE_X64             NI_SSE_X64_ConvertToInt64WithTruncation

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  SSE2 Intrinsics
#define FIRST_NI_SSE2               NI_SSE2_Add
HARDWARE_INTRINSIC(SSE2,            Add,                                        16,              2,     {INS_paddb,             INS_paddb,              INS_paddw,              INS_paddw,              INS_paddd,              INS_paddd,              INS_paddq,              INS_paddq,              INS_invalid,            INS_addpd},             HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            AddSaturate,                                16,              2,     {INS_paddsb,            INS_paddusb,            INS_paddsw,             INS_paddusw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            AddScalar,                                  16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_addsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2,            And,                                        16,              2,     {INS_pand,              INS_pand,               INS_pand,               INS_pand,               INS_pand,               INS_pand,               INS_pand,               INS_pand,               INS_invalid,            INS_andpd},             HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_NormalizeSmallTypeToInt)
HARDWARE_INTRINSIC(SSE2,            AndNot,                                     16,              2,     {INS_pandn,             INS_pandn,              INS_pandn,              INS_pandn,              INS_pandn,              INS_pandn,              INS_pandn,              INS_pandn,              INS_invalid,            INS_andnpd},            HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_NormalizeSmallTypeToInt)
HARDWARE_INTRINSIC(SSE2,            Average,                                    16,              2,     {INS_invalid,           INS_pavgb,              INS_invalid,            INS_pavgw,              INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            CompareEqual,                               16,              2,     {INS_pcmpeqb,           INS_pcmpeqb,            INS_pcmpeqw,            INS_pcmpeqw,            INS_pcmpeqd,            INS_pcmpeqd,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareGreaterThan,                         16,              2,     {INS_pcmpgtb,           INS_invalid,            INS_pcmpgtw,            INS_invalid,            INS_pcmpgtd,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareGreaterThanOrEqual,                  16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareLessThan,                            16,              2,     {INS_pcmpgtb,           INS_invalid,            INS_pcmpgtw,            INS_invalid,            INS_pcmpgtd,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareLessThanOrEqual,                     16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareNotEqual,                            16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareNotGreaterThan,                      16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareNotGreaterThanOrEqual,               16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareNotLessThan,                         16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareNotLessThanOrEqual,                  16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareOrdered,                             16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarEqual,                         16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarGreaterThan,                   16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpsd},             HW_Category_SIMDScalar,             HW_Flag_SpecialImport|HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarGreaterThanOrEqual,            16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpsd},             HW_Category_SIMDScalar,             HW_Flag_SpecialImport|HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarLessThan,                      16,              2,     {INS_pcmpgtb,           INS_invalid,            INS_pcmpgtw,            INS_invalid,            INS_pcmpgtd,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarLessThanOrEqual,               16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarNotEqual,                      16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarNotGreaterThan,                16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpsd},             HW_Category_SIMDScalar,             HW_Flag_SpecialImport|HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarNotGreaterThanOrEqual,         16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpsd},             HW_Category_SIMDScalar,             HW_Flag_SpecialImport|HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarNotLessThan,                   16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarNotLessThanOrEqual,            16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarOrdered,                       16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarOrderedEqual,                  16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_comisd},            HW_Category_SIMDScalar,             HW_Flag_Commutative|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarOrderedGreaterThan,            16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_comisd},            HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarOrderedGreaterThanOrEqual,     16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_comisd},            HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarOrderedLessThan,               16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_comisd},            HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarOrderedLessThanOrEqual,        16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_comisd},            HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarOrderedNotEqual,               16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_comisd},            HW_Category_SIMDScalar,             HW_Flag_Commutative|HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarUnordered,                     16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarUnorderedEqual,                16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_ucomisd},           HW_Category_SIMDScalar,             HW_Flag_Commutative|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarUnorderedGreaterThan,          16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_ucomisd},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarUnorderedGreaterThanOrEqual,   16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_ucomisd},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarUnorderedLessThan,             16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_ucomisd},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarUnorderedLessThanOrEqual,      16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_ucomisd},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareScalarUnorderedNotEqual,             16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_ucomisd},           HW_Category_SIMDScalar,             HW_Flag_Commutative|HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            CompareUnordered,                           16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            ConvertScalarToVector128Double,             16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtsi2sd32,         INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtss2sd,           INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(SSE2,            ConvertScalarToVector128Int32,              16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_movd,               INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            ConvertScalarToVector128Single,             16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtsd2ss,           INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE2,            ConvertScalarToVector128UInt32,             16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movd,               INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            ConvertToInt32,                             16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_movd,               INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtsd2si},          HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            ConvertToInt32WithTruncation,               16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvttsd2si32},       HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            ConvertToUInt32,                            16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movd,               INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            ConvertToVector128Double,                   16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtdq2pd,           INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtps2pd,           INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            ConvertToVector128Int32,                    16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtps2dq,           INS_cvtpd2dq},          HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            ConvertToVector128Int32WithTruncation,      16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvttps2dq,          INS_cvttpd2dq},         HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            ConvertToVector128Single,                   16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtdq2ps,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtpd2ps},          HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            Divide,                                     16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_divpd},             HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            DivideScalar,                               16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_divsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2,            Extract,                                    16,              2,     {INS_invalid,           INS_invalid,            INS_pextrw,             INS_pextrw,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            Insert,                                     16,              3,     {INS_invalid,           INS_invalid,            INS_pinsrw,             INS_pinsrw,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_CanBenefitFromConstantProp)
HARDWARE_INTRINSIC(SSE2,            LoadAlignedVector128,                       16,              1,     {INS_movdqa,            INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_invalid,            INS_movapd},            HW_Category_MemoryLoad,             HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            LoadFence,                                   0,              0,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment|HW_Flag_NoRMWSemantics|HW_Flag_SpecialSideEffect_Barrier)
HARDWARE_INTRINSIC(SSE2,            LoadHigh,                                   16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movhpd},            HW_Category_MemoryLoad,             HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            LoadLow,                                    16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movlpd},            HW_Category_MemoryLoad,             HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            LoadScalarVector128,                        16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_movd,               INS_movd,               INS_movq,               INS_movq,               INS_invalid,            INS_movsd_simd},        HW_Category_MemoryLoad,             HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            LoadVector128,                              16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(SSE2,            MaskMove,                                   16,              3,     {INS_maskmovdqu,        INS_maskmovdqu,         INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_MemoryStore,            HW_Flag_NoContainment|HW_Flag_NoRMWSemantics|HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(SSE2,            Max,                                        16,              2,     {INS_invalid,           INS_pmaxub,             INS_pmaxsw,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_maxpd},             HW_Category_SimpleSIMD,             HW_Flag_MaybeCommutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            MaxScalar,                                  16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_maxsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2,            MemoryFence,                                 0,              0,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment|HW_Flag_NoRMWSemantics|HW_Flag_SpecialSideEffect_Barrier)
HARDWARE_INTRINSIC(SSE2,            Min,                                        16,              2,     {INS_invalid,           INS_pminub,             INS_pminsw,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_minpd},             HW_Category_SimpleSIMD,             HW_Flag_MaybeCommutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            MinScalar,                                  16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_minsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2,            MoveMask,                                   16,              1,     {INS_pmovmskb,          INS_pmovmskb,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movmskpd},          HW_Category_SimpleSIMD,             HW_Flag_NoContainment|HW_Flag_NoRMWSemantics|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE2,            MoveScalar,                                 16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movq,               INS_movq,               INS_invalid,            INS_movsd_simd},        HW_Category_SIMDScalar,             HW_Flag_NoContainment)
HARDWARE_INTRINSIC(SSE2,            Multiply,                                   16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_pmuludq,            INS_invalid,            INS_mulpd},             HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            MultiplyAddAdjacent,                        16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_pmaddwd,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            MultiplyHigh,                               16,              2,     {INS_invalid,           INS_invalid,            INS_pmulhw,             INS_pmulhuw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            MultiplyLow,                                16,              2,     {INS_invalid,           INS_invalid,            INS_pmullw,             INS_pmullw,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            MultiplyScalar,                             16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_mulsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2,            Or,                                         16,              2,     {INS_por,               INS_por,                INS_por,                INS_por,                INS_por,                INS_por,                INS_por,                INS_por,                INS_invalid,            INS_orpd},              HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_NormalizeSmallTypeToInt)
HARDWARE_INTRINSIC(SSE2,            PackSignedSaturate,                         16,              2,     {INS_packsswb,          INS_invalid,            INS_packssdw,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            PackUnsignedSaturate,                       16,              2,     {INS_invalid,           INS_packuswb,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            ShiftLeftLogical,                           16,              2,     {INS_invalid,           INS_invalid,            INS_psllw,              INS_psllw,              INS_pslld,              INS_pslld,              INS_psllq,              INS_psllq,              INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_MaybeIMM|HW_Flag_NoJmpTableIMM|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            ShiftLeftLogical128BitLane,                 16,              2,     {INS_pslldq,            INS_pslldq,             INS_pslldq,             INS_pslldq,             INS_pslldq,             INS_pslldq,             INS_pslldq,             INS_pslldq,             INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(SSE2,            ShiftRightArithmetic,                       16,              2,     {INS_invalid,           INS_invalid,            INS_psraw,              INS_invalid,            INS_psrad,              INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_MaybeIMM|HW_Flag_NoJmpTableIMM|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            ShiftRightLogical,                          16,              2,     {INS_invalid,           INS_invalid,            INS_psrlw,              INS_psrlw,              INS_psrld,              INS_psrld,              INS_psrlq,              INS_psrlq,              INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_MaybeIMM|HW_Flag_NoJmpTableIMM|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            ShiftRightLogical128BitLane,                16,              2,     {INS_psrldq,            INS_psrldq,             INS_psrldq,             INS_psrldq,             INS_psrldq,             INS_psrldq,             INS_psrldq,             INS_psrldq,             INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(SSE2,            Shuffle,                                    16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_pshufd,             INS_pshufd,             INS_invalid,            INS_invalid,            INS_invalid,            INS_shufpd},            HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            ShuffleHigh,                                16,              2,     {INS_invalid,           INS_invalid,            INS_pshufhw,            INS_pshufhw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            ShuffleLow,                                 16,              2,     {INS_invalid,           INS_invalid,            INS_pshuflw,            INS_pshuflw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            Sqrt,                                       16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_sqrtpd},            HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            SqrtScalar,                                 16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_sqrtsd},            HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2,            Store,                                      16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(SSE2,            StoreAligned,                               16,              2,     {INS_movdqa,            INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_invalid,            INS_movapd},            HW_Category_MemoryStore,            HW_Flag_NoRMWSemantics|HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(SSE2,            StoreAlignedNonTemporal,                    16,              2,     {INS_movntdq,           INS_movntdq,            INS_movntdq,            INS_movntdq,            INS_movntdq,            INS_movntdq,            INS_movntdq,            INS_movntdq,            INS_invalid,            INS_movntpd},           HW_Category_MemoryStore,            HW_Flag_NoRMWSemantics|HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(SSE2,            StoreHigh,                                  16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movhpd},            HW_Category_MemoryStore,            HW_Flag_NoRMWSemantics|HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(SSE2,            StoreLow,                                   16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movlpd},            HW_Category_MemoryStore,            HW_Flag_NoRMWSemantics|HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(SSE2,            StoreNonTemporal,                           16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_movnti,             INS_movnti,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_MemoryStore,            HW_Flag_NoRMWSemantics|HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(SSE2,            StoreScalar,                                16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_movd,               INS_movd,               INS_movq,               INS_movq,               INS_invalid,            INS_movsd_simd},        HW_Category_MemoryStore,            HW_Flag_NoRMWSemantics|HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(SSE2,            Subtract,                                   16,              2,     {INS_psubb,             INS_psubb,              INS_psubw,              INS_psubw,              INS_psubd,              INS_psubd,              INS_psubq,              INS_psubq,              INS_invalid,            INS_subpd},             HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            SubtractSaturate,                           16,              2,     {INS_psubsb,            INS_psubusb,            INS_psubsw,             INS_psubusw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            SubtractScalar,                             16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_subsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE2,            SumAbsoluteDifferences,                     16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_psadbw,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(SSE2,            UnpackHigh,                                 16,              2,     {INS_punpckhbw,         INS_punpckhbw,          INS_punpckhwd,          INS_punpckhwd,          INS_punpckhdq,          INS_punpckhdq,          INS_punpckhqdq,         INS_punpckhqdq,         INS_invalid,            INS_unpckhpd},          HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            UnpackLow,                                  16,              2,     {INS_punpcklbw,         INS_punpcklbw,          INS_punpcklwd,          INS_punpcklwd,          INS_punpckldq,          INS_punpckldq,          INS_punpcklqdq,         INS_punpcklqdq,         INS_invalid,            INS_unpcklpd},          HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE2,            Xor,                                        16,              2,     {INS_pxor,              INS_pxor,               INS_pxor,               INS_pxor,               INS_pxor,               INS_pxor,               INS_pxor,               INS_pxor,               INS_invalid,            INS_xorpd},             HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_CanBenefitFromConstantProp|HW_Flag_NormalizeSmallTypeToInt)
#define LAST_NI_SSE2                NI_SSE2_Xor

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  SSE2 64-bit-only Intrinsics
#define FIRST_NI_SSE2_X64           NI_SSE2_X64_ConvertScalarToVector128Double
HARDWARE_INTRINSIC(SSE2_X64,        ConvertScalarToVector128Double,             16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtsi2sd64,         INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(SSE2_X64,        ConvertScalarToVector128Int64,              16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movd,               INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_NoRMWSemantics|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(SSE2_X64,        ConvertScalarToVector128UInt64,             16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movd,               INS_invalid,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_NoRMWSemantics|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(SSE2_X64,        ConvertToInt64,                             16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movd,               INS_invalid,            INS_invalid,            INS_cvtsd2si},          HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_X64,        ConvertToInt64WithTruncation,               16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvttsd2si64},         HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_X64,        ConvertToUInt64,                            16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movd,               INS_invalid,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2_X64,        StoreNonTemporal,                           16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movnti,             INS_movnti,             INS_invalid,            INS_invalid},           HW_Category_MemoryStore,            HW_Flag_NoRMWSemantics|HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromSecondArg)
#define LAST_NI_SSE2_X64            NI_SSE2_X64_StoreNonTemporal

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  SSE3 Intrinsics
#define FIRST_NI_SSE3               NI_SSE3_AddSubtract
HARDWARE_INTRINSIC(SSE3,            AddSubtract,                                16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_addsubps,           INS_addsubpd},          HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE3,            HorizontalAdd,                              16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_haddps,             INS_haddpd},            HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE3,            HorizontalSubtract,                         16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_hsubps,             INS_hsubpd},            HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE3,            LoadAndDuplicateToVector128,                16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_lddqu,              INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movddup},           HW_Category_MemoryLoad,             HW_Flag_NoRMWSemantics|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE3,            LoadDquVector128,                           16,              1,     {INS_lddqu,             INS_lddqu,              INS_lddqu,              INS_lddqu,              INS_lddqu,              INS_lddqu,              INS_lddqu,              INS_lddqu,              INS_invalid,            INS_invalid},           HW_Category_MemoryLoad,             HW_Flag_NoRMWSemantics|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE3,            MoveAndDuplicate,                           16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movddup},           HW_Category_SimpleSIMD,             HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE3,            MoveHighAndDuplicate,                       16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movshdup,           INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE3,            MoveLowAndDuplicate,                        16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movsldup,           INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoRMWSemantics)
#define LAST_NI_SSE3                NI_SSE3_MoveLowAndDuplicate

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  SSSE3 Intrinsics
#define FIRST_NI_SSSE3              NI_SSSE3_Abs
HARDWARE_INTRINSIC(SSSE3,           Abs,                                        16,              1,     {INS_pabsb,             INS_invalid,            INS_pabsw,              INS_invalid,            INS_pabsd,              INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSSE3,           AlignRight,                                 16,              3,     {INS_palignr,           INS_palignr,            INS_palignr,            INS_palignr,            INS_palignr,            INS_palignr,            INS_palignr,            INS_palignr,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSSE3,           HorizontalAdd,                              16,              2,     {INS_invalid,           INS_invalid,            INS_phaddw,             INS_phaddw,             INS_phaddd,             INS_phaddd,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSSE3,           HorizontalAddSaturate,                      16,              2,     {INS_invalid,           INS_invalid,            INS_phaddsw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSSE3,           HorizontalSubtract,                         16,              2,     {INS_invalid,           INS_invalid,            INS_phsubw,             INS_invalid,            INS_phsubd,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSSE3,           HorizontalSubtractSaturate,                 16,              2,     {INS_invalid,           INS_invalid,            INS_phsubsw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSSE3,           MultiplyAddAdjacent,                        16,              2,     {INS_invalid,           INS_invalid,            INS_pmaddubsw,          INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSSE3,           MultiplyHighRoundScale,                     16,              2,     {INS_invalid,           INS_invalid,            INS_pmulhrsw,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSSE3,           Shuffle,                                    16,              2,     {INS_pshufb,            INS_pshufb,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSSE3,           Sign,                                       16,              2,     {INS_psignb,            INS_invalid,            INS_psignw,             INS_invalid,            INS_psignd,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
#define LAST_NI_SSSE3               NI_SSSE3_Sign

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  SSE41 Intrinsics
#define FIRST_NI_SSE41              NI_SSE41_Blend
HARDWARE_INTRINSIC(SSE41,           Blend,                                      16,              3,     {INS_invalid,           INS_invalid,            INS_pblendw,            INS_pblendw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_blendps,            INS_blendpd},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE41,           BlendVariable,                              16,              3,     {INS_pblendvb,          INS_pblendvb,           INS_pblendvb,           INS_pblendvb,           INS_pblendvb,           INS_pblendvb,           INS_pblendvb,           INS_pblendvb,           INS_blendvps,           INS_blendvpd},          HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE41,           Ceiling,                                    16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundps,            INS_roundpd},           HW_Category_SimpleSIMD,             HW_Flag_NoRMWSemantics|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE41,           CeilingScalar,                              16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundss,            INS_roundsd},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE41,           CompareEqual,                               16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_pcmpeqq,            INS_pcmpeqq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE41,           ConvertToVector128Int16,                    16,              1,     {INS_pmovsxbw,          INS_pmovzxbw,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics|HW_Flag_MaybeMemoryLoad)
HARDWARE_INTRINSIC(SSE41,           ConvertToVector128Int32,                    16,              1,     {INS_pmovsxbd,          INS_pmovzxbd,           INS_pmovsxwd,           INS_pmovzxwd,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics|HW_Flag_MaybeMemoryLoad)
HARDWARE_INTRINSIC(SSE41,           ConvertToVector128Int64,                    16,              1,     {INS_pmovsxbq,          INS_pmovzxbq,           INS_pmovsxwq,           INS_pmovzxwq,           INS_pmovsxdq,           INS_pmovzxdq,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoRMWSemantics|HW_Flag_MaybeMemoryLoad)
HARDWARE_INTRINSIC(SSE41,           DotProduct,                                 16,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_dpps,               INS_dppd},              HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE41,           Extract,                                    16,              2,     {INS_pextrb,            INS_pextrb,             INS_invalid,            INS_invalid,            INS_pextrd,             INS_pextrd,             INS_invalid,            INS_invalid,            INS_extractps,          INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE41,           Floor,                                      16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundps,            INS_roundpd},           HW_Category_SimpleSIMD,             HW_Flag_NoRMWSemantics|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE41,           FloorScalar,                                16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundss,            INS_roundsd},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE41,           Insert,                                     16,              3,     {INS_pinsrb,            INS_pinsrb,             INS_invalid,            INS_invalid,            INS_pinsrd,             INS_pinsrd,             INS_invalid,            INS_invalid,            INS_insertps,           INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_CanBenefitFromConstantProp)
HARDWARE_INTRINSIC(SSE41,           LoadAlignedVector128NonTemporal,            16,              1,     {INS_movntdqa,          INS_movntdqa,           INS_movntdqa,           INS_movntdqa,           INS_movntdqa,           INS_movntdqa,           INS_movntdqa,           INS_movntdqa,           INS_invalid,            INS_invalid},           HW_Category_MemoryLoad,             HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE41,           Max,                                        16,              2,     {INS_pmaxsb,            INS_invalid,            INS_invalid,            INS_pmaxuw,             INS_pmaxsd,             INS_pmaxud,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE41,           Min,                                        16,              2,     {INS_pminsb,            INS_invalid,            INS_invalid,            INS_pminuw,             INS_pminsd,             INS_pminud,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_Commutative)
HARDWARE_INTRINSIC(SSE41,           MinHorizontal,                              16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_phminposuw,         INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoRMWSemantics|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE41,           MultipleSumAbsoluteDifferences,             16,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_mpsadbw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE41,           Multiply,                                   16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_pmuldq,             INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE41,           MultiplyLow,                                16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_pmulld,             INS_pmulld,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE41,           PackUnsignedSaturate,                       16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_packusdw,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE41,           RoundCurrentDirection,                      16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundps,            INS_roundpd},           HW_Category_SimpleSIMD,             HW_Flag_NoRMWSemantics|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE41,           RoundCurrentDirectionScalar,                16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundss,            INS_roundsd},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE41,           RoundToNearestInteger,                      16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundps,            INS_roundpd},           HW_Category_SimpleSIMD,             HW_Flag_NoRMWSemantics|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE41,           RoundToNearestIntegerScalar,                16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundss,            INS_roundsd},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE41,           RoundToNegativeInfinity,                    16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundps,            INS_roundpd},           HW_Category_SimpleSIMD,             HW_Flag_NoRMWSemantics|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE41,           RoundToNegativeInfinityScalar,              16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundss,            INS_roundsd},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE41,           RoundToPositiveInfinity,                    16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundps,            INS_roundpd},           HW_Category_SimpleSIMD,             HW_Flag_NoRMWSemantics|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE41,           RoundToPositiveInfinityScalar,              16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundss,            INS_roundsd},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE41,           RoundToZero,                                16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundps,            INS_roundpd},           HW_Category_SimpleSIMD,             HW_Flag_NoRMWSemantics|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(SSE41,           RoundToZeroScalar,                          16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundss,            INS_roundsd},           HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(SSE41,           TestC,                                      16,              2,     {INS_ptest,             INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE41,           TestNotZAndNotC,                            16,              2,     {INS_ptest,             INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE41,           TestZ,                                      16,              2,     {INS_ptest,             INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoEvexSemantics)
#define LAST_NI_SSE41               NI_SSE41_TestZ

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  SSE41 64-bit-only Intrinsics
#define FIRST_NI_SSE41_X64          NI_SSE41_X64_Extract
HARDWARE_INTRINSIC(SSE41_X64,       Extract,                                    16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_pextrq,             INS_pextrq,             INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE41_X64,       Insert,                                     16,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_pinsrq,             INS_pinsrq,             INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_CanBenefitFromConstantProp)
#define LAST_NI_SSE41_X64          NI_SSE41_X64_Insert

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  SSE42 Intrinsics
#define FIRST_NI_SSE42              NI_SSE42_CompareGreaterThan
HARDWARE_INTRINSIC(SSE42,           CompareGreaterThan,                         16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_pcmpgtq,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE42,           CompareLessThan,                            16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_pcmpgtq,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(SSE42,           Crc32,                                       0,              2,     {INS_invalid,           INS_crc32,              INS_invalid,            INS_crc32,              INS_invalid,            INS_crc32,              INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_RmwIntrinsic)
#define LAST_NI_SSE42               NI_SSE42_Crc32

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  SSE42 Intrinsics
#define FIRST_NI_SSE42_X64          NI_SSE42_X64_Crc32
HARDWARE_INTRINSIC(SSE42_X64,       Crc32,                                       0,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_crc32,              INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_RmwIntrinsic)
#define LAST_NI_SSE42_X64           NI_SSE42_X64_Crc32

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  AVX Intrinsics
#define FIRST_NI_AVX                NI_AVX_Add
HARDWARE_INTRINSIC(AVX,             Add,                                        32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_addps,              INS_addpd},             HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             AddSubtract,                                32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_addsubps,           INS_addsubpd},          HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             And,                                        32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_andps,              INS_andpd},             HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             AndNot,                                     32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_andnps,             INS_andnpd},            HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             Blend,                                      32,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_blendps,            INS_blendpd},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             BlendVariable,                              32,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vblendvps,          INS_vblendvpd},         HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             BroadcastScalarToVector128,                 16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vbroadcastss,       INS_invalid},           HW_Category_MemoryLoad,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX,             BroadcastScalarToVector256,                 32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vbroadcastss,       INS_vbroadcastsd},      HW_Category_MemoryLoad,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX,             BroadcastVector128ToVector256,              32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vbroadcastf128,     INS_vbroadcastf128},    HW_Category_MemoryLoad,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX,             Ceiling,                                    32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundps,            INS_roundpd},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             Compare,                                    32,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_cmppd},             HW_Category_IMM,                    HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics|HW_Flag_SpecialImport)
HARDWARE_INTRINSIC(AVX,             CompareEqual,                               32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             CompareGreaterThan,                         32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             CompareGreaterThanOrEqual,                  32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             CompareLessThan,                            32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             CompareLessThanOrEqual,                     32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             CompareNotEqual,                            32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             CompareNotGreaterThan,                      32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             CompareNotGreaterThanOrEqual,               32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             CompareNotLessThan,                         32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             CompareNotLessThanOrEqual,                  32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             CompareOrdered,                             32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             CompareScalar,                              16,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpss,              INS_cmpsd},             HW_Category_IMM,                    HW_Flag_CopyUpperBits|HW_Flag_NoEvexSemantics|HW_Flag_SpecialImport)
HARDWARE_INTRINSIC(AVX,             CompareUnordered,                           32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cmpps,              INS_cmppd},             HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             ConvertToVector128Int32,                    32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtpd2dq,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtpd2dq},          HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             ConvertToVector128Int32WithTruncation,      32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_cvttpd2dq,          INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             ConvertToVector128Single,                   32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtpd2ps},          HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             ConvertToVector256Double,                   32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtdq2pd,           INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtps2pd,           INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             ConvertToVector256Int32,                    32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtps2dq,           INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtps2dq,           INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             ConvertToVector256Int32WithTruncation,      32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvttps2dq,          INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(AVX,             ConvertToVector256Single,                   32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtdq2ps,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             Divide,                                     32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_divps,              INS_divpd},             HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             DotProduct,                                 32,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_dpps,               INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             DuplicateEvenIndexed,                       32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movsldup,           INS_movddup},           HW_Category_SimpleSIMD,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX,             DuplicateOddIndexed,                        32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movshdup,           INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX,             ExtractVector128,                           32,              2,     {INS_vextractf128,      INS_vextractf128,       INS_vextractf128,       INS_vextractf128,       INS_vextractf128,       INS_vextractf128,       INS_vextractf128,       INS_vextractf128,       INS_vextractf128,       INS_vextractf128},      HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             Floor,                                      32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundps,            INS_roundpd},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             HorizontalAdd,                              32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_haddps,             INS_haddpd},            HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             HorizontalSubtract,                         32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_hsubps,             INS_hsubpd},            HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             InsertVector128,                            32,              3,     {INS_vinsertf128,       INS_vinsertf128,        INS_vinsertf128,        INS_vinsertf128,        INS_vinsertf128,        INS_vinsertf128,        INS_vinsertf128,        INS_vinsertf128,        INS_vinsertf128,        INS_vinsertf128},       HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             LoadAlignedVector256,                       32,              1,     {INS_movdqa,            INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movaps,             INS_movapd},            HW_Category_MemoryLoad,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX,             LoadDquVector256,                           32,              1,     {INS_lddqu,             INS_lddqu,              INS_lddqu,              INS_lddqu,              INS_lddqu,              INS_lddqu,              INS_lddqu,              INS_lddqu,              INS_invalid,            INS_invalid},           HW_Category_MemoryLoad,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             LoadVector256,                              32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX,             MaskLoad,                                   -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vmaskmovps,         INS_vmaskmovpd},        HW_Category_MemoryLoad,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             MaskStore,                                  -1,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vmaskmovps,         INS_vmaskmovpd},        HW_Category_MemoryStore,            HW_Flag_NoContainment|HW_Flag_BaseTypeFromSecondArg|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             Max,                                        32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_maxps,              INS_maxpd},             HW_Category_SimpleSIMD,             HW_Flag_MaybeCommutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             Min,                                        32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_minps,              INS_minpd},             HW_Category_SimpleSIMD,             HW_Flag_MaybeCommutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             MoveMask,                                   32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movmskps,           INS_movmskpd},          HW_Category_SimpleSIMD,             HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             Multiply,                                   32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_mulps,              INS_mulpd},             HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             Or,                                         32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_orps,               INS_orpd},              HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             Permute,                                    -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermilps,          INS_vpermilpd},         HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             Permute2x128,                               32,              3,     {INS_vperm2f128,        INS_vperm2f128,         INS_vperm2f128,         INS_vperm2f128,         INS_vperm2f128,         INS_vperm2f128,         INS_vperm2f128,         INS_vperm2f128,         INS_vperm2f128,         INS_vperm2f128},        HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             PermuteVar,                                 -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermilpsvar,       INS_vpermilpdvar},      HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             Reciprocal,                                 32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_rcpps,              INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             ReciprocalSqrt,                             32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_rsqrtps,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             RoundCurrentDirection,                      32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundps,            INS_roundpd},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             RoundToNearestInteger,                      32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundps,            INS_roundpd},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             RoundToNegativeInfinity,                    32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundps,            INS_roundpd},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             RoundToPositiveInfinity,                    32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundps,            INS_roundpd},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             RoundToZero,                                32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_roundps,            INS_roundpd},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             Shuffle,                                    32,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_shufps,             INS_shufpd},            HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             Sqrt,                                       32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_sqrtps,             INS_sqrtpd},            HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             Store,                                      32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(AVX,             StoreAligned,                               32,              2,     {INS_movdqa,            INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movaps,             INS_movapd},            HW_Category_MemoryStore,            HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(AVX,             StoreAlignedNonTemporal,                    32,              2,     {INS_movntdq,           INS_movntdq,            INS_movntdq,            INS_movntdq,            INS_movntdq,            INS_movntdq,            INS_movntdq,            INS_movntdq,            INS_movntps,            INS_movntpd},           HW_Category_MemoryStore,            HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(AVX,             Subtract,                                   32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_subps,              INS_subpd},             HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             TestC,                                      -1,              2,     {INS_ptest,             INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_vtestps,            INS_vtestpd},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             TestNotZAndNotC,                            -1,              2,     {INS_ptest,             INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_vtestps,            INS_vtestpd},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             TestZ,                                      -1,              2,     {INS_ptest,             INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_vtestps,            INS_vtestpd},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             UnpackHigh,                                 32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_unpckhps,           INS_unpckhpd},          HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             UnpackLow,                                  32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_unpcklps,           INS_unpcklpd},          HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX,             Xor,                                        32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_xorps,              INS_xorpd},             HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_CanBenefitFromConstantProp)
#define LAST_NI_AVX                 NI_AVX_Xor

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  AVX2 Intrinsics
#define FIRST_NI_AVX2               NI_AVX2_Abs
HARDWARE_INTRINSIC(AVX2,            Abs,                                        32,              1,     {INS_pabsb,             INS_invalid,            INS_pabsw,              INS_invalid,            INS_pabsd,              INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            Add,                                        32,              2,     {INS_paddb,             INS_paddb,              INS_paddw,              INS_paddw,              INS_paddd,              INS_paddd,              INS_paddq,              INS_paddq,              INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            AddSaturate,                                32,              2,     {INS_paddsb,            INS_paddusb,            INS_paddsw,             INS_paddusw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            AlignRight,                                 32,              3,     {INS_palignr,           INS_palignr,            INS_palignr,            INS_palignr,            INS_palignr,            INS_palignr,            INS_palignr,            INS_palignr,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            And,                                        32,              2,     {INS_pand,              INS_pand,               INS_pand,               INS_pand,               INS_pand,               INS_pand,               INS_pand,               INS_pand,               INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_NormalizeSmallTypeToInt)
HARDWARE_INTRINSIC(AVX2,            AndNot,                                     32,              2,     {INS_pandn,             INS_pandn,              INS_pandn,              INS_pandn,              INS_pandn,              INS_pandn,              INS_pandn,              INS_pandn,              INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_NormalizeSmallTypeToInt)
HARDWARE_INTRINSIC(AVX2,            Average,                                    32,              2,     {INS_invalid,           INS_pavgb,              INS_invalid,            INS_pavgw,              INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            Blend,                                      -1,              3,     {INS_invalid,           INS_invalid,            INS_pblendw,            INS_pblendw,            INS_vpblendd,           INS_vpblendd,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX2,            BlendVariable,                              32,              3,     {INS_vpblendvb,         INS_vpblendvb,          INS_vpblendvb,          INS_vpblendvb,          INS_vpblendvb,          INS_vpblendvb,          INS_vpblendvb,          INS_vpblendvb,          INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX2,            BroadcastScalarToVector128,                 16,              1,     {INS_vpbroadcastb,      INS_vpbroadcastb,       INS_vpbroadcastw,       INS_vpbroadcastw,       INS_vpbroadcastd,       INS_vpbroadcastd,       INS_vpbroadcastq,       INS_vpbroadcastq,       INS_vbroadcastss,       INS_movddup},           HW_Category_SIMDScalar,             HW_Flag_MaybeMemoryLoad)
HARDWARE_INTRINSIC(AVX2,            BroadcastScalarToVector256,                 32,              1,     {INS_vpbroadcastb,      INS_vpbroadcastb,       INS_vpbroadcastw,       INS_vpbroadcastw,       INS_vpbroadcastd,       INS_vpbroadcastd,       INS_vpbroadcastq,       INS_vpbroadcastq,       INS_vbroadcastss,       INS_vbroadcastsd},      HW_Category_SIMDScalar,             HW_Flag_MaybeMemoryLoad)
HARDWARE_INTRINSIC(AVX2,            BroadcastVector128ToVector256,              32,              1,     {INS_vbroadcasti128,    INS_vbroadcasti128,     INS_vbroadcasti128,     INS_vbroadcasti128,     INS_vbroadcasti128,     INS_vbroadcasti128,     INS_vbroadcasti128,     INS_vbroadcasti128,     INS_invalid,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_MaybeMemoryLoad)
HARDWARE_INTRINSIC(AVX2,            CompareEqual,                               32,              2,     {INS_pcmpeqb,           INS_pcmpeqb,            INS_pcmpeqw,            INS_pcmpeqw,            INS_pcmpeqd,            INS_pcmpeqd,            INS_pcmpeqq,            INS_pcmpeqq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX2,            CompareGreaterThan,                         32,              2,     {INS_pcmpgtb,           INS_invalid,            INS_pcmpgtw,            INS_invalid,            INS_pcmpgtd,            INS_invalid,            INS_pcmpgtq,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX2,            CompareLessThan,                            32,              2,     {INS_pcmpgtb,           INS_invalid,            INS_pcmpgtw,            INS_invalid,            INS_pcmpgtd,            INS_invalid,            INS_pcmpgtq,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX2,            ConvertToInt32,                             32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_movd,               INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX2,            ConvertToUInt32,                            32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movd,               INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX2,            ConvertToVector256Int16,                    32,              1,     {INS_pmovsxbw,          INS_pmovzxbw,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromFirstArg|HW_Flag_MaybeMemoryLoad)
HARDWARE_INTRINSIC(AVX2,            ConvertToVector256Int32,                    32,              1,     {INS_pmovsxbd,          INS_pmovzxbd,           INS_pmovsxwd,           INS_pmovzxwd,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromFirstArg|HW_Flag_MaybeMemoryLoad)
HARDWARE_INTRINSIC(AVX2,            ConvertToVector256Int64,                    32,              1,     {INS_pmovsxbq,          INS_pmovzxbq,           INS_pmovsxwq,           INS_pmovzxwq,           INS_pmovsxdq,           INS_pmovzxdq,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_BaseTypeFromFirstArg|HW_Flag_MaybeMemoryLoad)
HARDWARE_INTRINSIC(AVX2,            ExtractVector128,                           32,              2,     {INS_vextracti128,      INS_vextracti128,       INS_vextracti128,       INS_vextracti128,       INS_vextracti128,       INS_vextracti128,       INS_vextracti128,       INS_vextracti128,       INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            GatherMaskVector128,                        16,              5,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpgatherdd,         INS_vpgatherdd,         INS_vpgatherdq,         INS_vpgatherdq,         INS_vgatherdps,         INS_vgatherdpd},        HW_Category_IMM,                    HW_Flag_MaybeMemoryLoad|HW_Flag_SpecialCodeGen|HW_Flag_SpecialImport|HW_Flag_NoContainment|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX2,            GatherMaskVector256,                        32,              5,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpgatherdd,         INS_vpgatherdd,         INS_vpgatherdq,         INS_vpgatherdq,         INS_vgatherdps,         INS_vgatherdpd},        HW_Category_IMM,                    HW_Flag_MaybeMemoryLoad|HW_Flag_SpecialCodeGen|HW_Flag_SpecialImport|HW_Flag_NoContainment|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX2,            GatherVector128,                            16,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpgatherdd,         INS_vpgatherdd,         INS_vpgatherdq,         INS_vpgatherdq,         INS_vgatherdps,         INS_vgatherdpd},        HW_Category_IMM,                    HW_Flag_MaybeMemoryLoad|HW_Flag_SpecialCodeGen|HW_Flag_NoContainment|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX2,            GatherVector256,                            32,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpgatherdd,         INS_vpgatherdd,         INS_vpgatherdq,         INS_vpgatherdq,         INS_vgatherdps,         INS_vgatherdpd},        HW_Category_IMM,                    HW_Flag_MaybeMemoryLoad|HW_Flag_SpecialCodeGen|HW_Flag_NoContainment|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX2,            HorizontalAdd,                              32,              2,     {INS_invalid,           INS_invalid,            INS_phaddw,             INS_phaddw,             INS_phaddd,             INS_phaddd,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX2,            HorizontalAddSaturate,                      32,              2,     {INS_invalid,           INS_invalid,            INS_phaddsw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX2,            HorizontalSubtract,                         32,              2,     {INS_invalid,           INS_invalid,            INS_phsubw,             INS_invalid,            INS_phsubd,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX2,            HorizontalSubtractSaturate,                 32,              2,     {INS_invalid,           INS_invalid,            INS_phsubsw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX2,            InsertVector128,                            32,              3,     {INS_vinserti128,       INS_vinserti128,        INS_vinserti128,        INS_vinserti128,        INS_vinserti128,        INS_vinserti128,        INS_vinserti128,        INS_vinserti128,        INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            LoadAlignedVector256NonTemporal,            32,              1,     {INS_movntdqa,          INS_movntdqa,           INS_movntdqa,           INS_movntdqa,           INS_movntdqa,           INS_movntdqa,           INS_movntdqa,           INS_movntdqa,           INS_invalid,            INS_invalid},           HW_Category_MemoryLoad,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX2,            MaskLoad,                                   -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmaskmovd,         INS_vpmaskmovd,         INS_vpmaskmovq,         INS_vpmaskmovq,         INS_invalid,            INS_invalid},           HW_Category_MemoryLoad,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX2,            MaskStore,                                  -1,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmaskmovd,         INS_vpmaskmovd,         INS_vpmaskmovq,         INS_vpmaskmovq,         INS_invalid,            INS_invalid},           HW_Category_MemoryStore,            HW_Flag_NoContainment|HW_Flag_BaseTypeFromSecondArg|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX2,            Max,                                        32,              2,     {INS_pmaxsb,            INS_pmaxub,             INS_pmaxsw,             INS_pmaxuw,             INS_pmaxsd,             INS_pmaxud,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX2,            Min,                                        32,              2,     {INS_pminsb,            INS_pminub,             INS_pminsw,             INS_pminuw,             INS_pminsd,             INS_pminud,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX2,            MoveMask,                                   32,              1,     {INS_pmovmskb,          INS_pmovmskb,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoContainment|HW_Flag_BaseTypeFromFirstArg|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX2,            MultipleSumAbsoluteDifferences,             32,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_mpsadbw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX2,            Multiply,                                   32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_pmuldq,             INS_pmuludq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            MultiplyAddAdjacent,                        32,              2,     {INS_invalid,           INS_invalid,            INS_pmaddubsw,          INS_invalid,            INS_pmaddwd,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            MultiplyHigh,                               32,              2,     {INS_invalid,           INS_invalid,            INS_pmulhw,             INS_pmulhuw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            MultiplyHighRoundScale,                     32,              2,     {INS_invalid,           INS_invalid,            INS_pmulhrsw,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            MultiplyLow,                                32,              2,     {INS_invalid,           INS_invalid,            INS_pmullw,             INS_pmullw,             INS_pmulld,             INS_pmulld,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            Or,                                         32,              2,     {INS_por,               INS_por,                INS_por,                INS_por,                INS_por,                INS_por,                INS_por,                INS_por,                INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_NormalizeSmallTypeToInt)
HARDWARE_INTRINSIC(AVX2,            PackSignedSaturate,                         32,              2,     {INS_packsswb,          INS_invalid,            INS_packssdw,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            PackUnsignedSaturate,                       32,              2,     {INS_invalid,           INS_packuswb,           INS_invalid,            INS_packusdw,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            Permute2x128,                               32,              3,     {INS_vperm2i128,        INS_vperm2i128,         INS_vperm2i128,         INS_vperm2i128,         INS_vperm2i128,         INS_vperm2i128,         INS_vperm2i128,         INS_vperm2i128,         INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX2,            Permute4x64,                                32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermq,             INS_vpermq,             INS_invalid,            INS_vpermpd},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            PermuteVar8x32,                             32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermd,             INS_vpermd,             INS_invalid,            INS_invalid,            INS_vpermps,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            ShiftLeftLogical,                           32,              2,     {INS_invalid,           INS_invalid,            INS_psllw,              INS_psllw,              INS_pslld,              INS_pslld,              INS_psllq,              INS_psllq,              INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_MaybeIMM|HW_Flag_NoJmpTableIMM|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            ShiftLeftLogical128BitLane,                 32,              2,     {INS_pslldq,            INS_pslldq,             INS_pslldq,             INS_pslldq,             INS_pslldq,             INS_pslldq,             INS_pslldq,             INS_pslldq,             INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(AVX2,            ShiftLeftLogicalVariable,                   -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpsllvd,            INS_vpsllvd,            INS_vpsllvq,            INS_vpsllvq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            ShiftRightArithmetic,                       32,              2,     {INS_invalid,           INS_invalid,            INS_psraw,              INS_invalid,            INS_psrad,              INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_MaybeIMM|HW_Flag_NoJmpTableIMM|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            ShiftRightArithmeticVariable,               -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpsravd,            INS_vpsravd,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            ShiftRightLogical,                          32,              2,     {INS_invalid,           INS_invalid,            INS_psrlw,              INS_psrlw,              INS_psrld,              INS_psrld,              INS_psrlq,              INS_psrlq,              INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_MaybeIMM|HW_Flag_NoJmpTableIMM|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            ShiftRightLogical128BitLane,                32,              2,     {INS_psrldq,            INS_psrldq,             INS_psrldq,             INS_psrldq,             INS_psrldq,             INS_psrldq,             INS_psrldq,             INS_psrldq,             INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(AVX2,            ShiftRightLogicalVariable,                  -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpsrlvd,            INS_vpsrlvd,            INS_vpsrlvq,            INS_vpsrlvq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            Shuffle,                                    32,              2,     {INS_pshufb,            INS_pshufb,             INS_invalid,            INS_invalid,            INS_pshufd,             INS_pshufd,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_MaybeIMM)
HARDWARE_INTRINSIC(AVX2,            ShuffleHigh,                                32,              2,     {INS_invalid,           INS_invalid,            INS_pshufhw,            INS_pshufhw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            ShuffleLow,                                 32,              2,     {INS_invalid,           INS_invalid,            INS_pshuflw,            INS_pshuflw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            Sign,                                       32,              2,     {INS_psignb,            INS_invalid,            INS_psignw,             INS_invalid,            INS_psignd,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX2,            Subtract,                                   32,              2,     {INS_psubb,             INS_psubb,              INS_psubw,              INS_psubw,              INS_psubd,              INS_psubd,              INS_psubq,              INS_psubq,              INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            SubtractSaturate,                           32,              2,     {INS_psubsb,            INS_psubusb,            INS_psubsw,             INS_psubusw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            SumAbsoluteDifferences,                     32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_psadbw,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX2,            UnpackHigh,                                 32,              2,     {INS_punpckhbw,         INS_punpckhbw,          INS_punpckhwd,          INS_punpckhwd,          INS_punpckhdq,          INS_punpckhdq,          INS_punpckhqdq,         INS_punpckhqdq,         INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            UnpackLow,                                  32,              2,     {INS_punpcklbw,         INS_punpcklbw,          INS_punpcklwd,          INS_punpcklwd,          INS_punpckldq,          INS_punpckldq,          INS_punpcklqdq,         INS_punpcklqdq,         INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX2,            Xor,                                        32,              2,     {INS_pxor,              INS_pxor,               INS_pxor,               INS_pxor,               INS_pxor,               INS_pxor,               INS_pxor,               INS_pxor,               INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_CanBenefitFromConstantProp|HW_Flag_NormalizeSmallTypeToInt)
#define LAST_NI_AVX2                NI_AVX2_Xor

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  AVX512F Intrinsics
#define FIRST_NI_AVX512F             NI_AVX512F_Abs
HARDWARE_INTRINSIC(AVX512F,         Abs,                                        64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_pabsd,              INS_invalid,            INS_vpabsq,             INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         Add,                                        64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_paddd,              INS_paddd,              INS_paddq,              INS_paddq,              INS_addps,              INS_addpd},             HW_Category_SimpleSIMD,             HW_Flag_MaybeCommutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         AddScalar,                                  16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_addss,              INS_addsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         AlignRight32,                               64,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_valignd,            INS_valignd,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         AlignRight64,                               64,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_valignq,            INS_valignq,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         And,                                        64,              2,     {INS_pand,              INS_pand,               INS_pand,               INS_pand,               INS_pand,               INS_pand,               INS_vpandq,             INS_vpandq,             INS_andps,              INS_andpd},             HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_NormalizeSmallTypeToInt)
HARDWARE_INTRINSIC(AVX512F,         AndNot,                                     64,              2,     {INS_pandn,             INS_pandn,              INS_pandn,              INS_pandn,              INS_pandn,              INS_pandn,              INS_vpandnq,            INS_vpandnq,            INS_andnps,             INS_andnpd},            HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_NormalizeSmallTypeToInt)
HARDWARE_INTRINSIC(AVX512F,         BlendVariable,                              64,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512F,         BroadcastScalarToVector512,                 64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpbroadcastd,       INS_vpbroadcastd,       INS_vpbroadcastq,       INS_vpbroadcastq,       INS_vbroadcastss,       INS_vbroadcastsd},      HW_Category_SIMDScalar,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX512F,         BroadcastVector128ToVector512,              64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vbroadcasti128,     INS_vbroadcasti128,     INS_invalid,            INS_invalid,            INS_vbroadcastf128,     INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_MaybeMemoryLoad)
HARDWARE_INTRINSIC(AVX512F,         BroadcastVector256ToVector512,              64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vbroadcasti64x4,    INS_vbroadcasti64x4,    INS_invalid,            INS_vbroadcastf64x4},   HW_Category_SimpleSIMD,             HW_Flag_MaybeMemoryLoad)
HARDWARE_INTRINSIC(AVX512F,         Compare,                                    64,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512F,         CompareEqual,                               64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512F,         CompareGreaterThan,                         64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512F,         CompareGreaterThanOrEqual,                  64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512F,         CompareLessThan,                            64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512F,         CompareLessThanOrEqual,                     64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512F,         CompareNotEqual,                            64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512F,         CompareNotGreaterThan,                      64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512F,         CompareNotGreaterThanOrEqual,               64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512F,         CompareNotLessThan,                         64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512F,         CompareNotLessThanOrEqual,                  64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512F,         CompareOrdered,                             64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512F,         CompareUnordered,                           64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512F,         ConvertScalarToVector128Double,             16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtsi2sd32,         INS_vcvtusi2sd32,       INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromSecondArg|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX512F,         ConvertScalarToVector128Single,             16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtsi2ss32,         INS_vcvtusi2ss32,       INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtsd2ss},          HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromSecondArg|HW_Flag_CopyUpperBits|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ConvertToInt32,                             16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtss2si,           INS_cvtsd2si},          HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ConvertToUInt32,                            16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtss2usi,         INS_vcvtsd2usi},        HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ConvertToUInt32WithTruncation,              16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttss2usi32,      INS_vcvttsd2usi32},     HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector128Byte,                     64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovdb,            INS_vpmovdb,            INS_vpmovqb,            INS_vpmovqb,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector128ByteWithSaturation,       64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovusdb,          INS_invalid,            INS_vpmovusqb,          INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector128Int16,                    64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovqw,            INS_vpmovqw,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector128Int16WithSaturation,      64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovsqw,           INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector128SByte,                    64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovdb,            INS_vpmovdb,            INS_vpmovqb,            INS_vpmovqb,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector128SByteWithSaturation,      64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovsdb,           INS_invalid,            INS_vpmovsqb,           INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector128UInt16,                   64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovqw,            INS_vpmovqw,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector128UInt16WithSaturation,     64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovusqw,          INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector256Int16,                    64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovdw,            INS_vpmovdw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector256Int16WithSaturation,      64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovsdw,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector256Int32,                    64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovqd,            INS_vpmovqd,            INS_invalid,            INS_cvtpd2dq},          HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector256Int32WithSaturation,      64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovsqd,           INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector256Int32WithTruncation,      64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvttpd2dq},         HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector256Single,                   64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtpd2ps},          HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector256UInt16,                   64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovdw,            INS_vpmovdw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector256UInt16WithSaturation,     64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovusdw,          INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector256UInt32,                   64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovqd,            INS_vpmovqd,            INS_invalid,            INS_vcvtpd2udq},        HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector256UInt32WithSaturation,     64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovusqd,          INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector256UInt32WithTruncation,     64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttpd2udq},       HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector512Double,                   64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtdq2pd,           INS_vcvtudq2pd,         INS_invalid,            INS_invalid,            INS_cvtps2pd,           INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector512Int32,                    64,             -1,     {INS_pmovsxbd,          INS_pmovzxbd,           INS_pmovsxwd,           INS_pmovzxwd,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtps2dq,           INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector512Int32WithTruncation,      64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvttps2dq,          INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector512Int64,                    64,              1,     {INS_pmovsxbq,          INS_pmovzxbq,           INS_pmovsxwq,           INS_pmovzxwq,           INS_pmovsxdq,           INS_pmovzxdq,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector512Single,                   64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtdq2ps,           INS_vcvtudq2ps,         INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector512UInt32,                   64,             -1,     {INS_pmovsxbd,          INS_pmovzxbd,           INS_pmovsxwd,           INS_pmovzxwd,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtps2udq,         INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector512UInt32WithTruncation,     64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttps2udq,        INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ConvertToVector512UInt64,                   64,              1,     {INS_pmovsxbq,          INS_pmovzxbq,           INS_pmovsxwq,           INS_pmovzxwq,           INS_pmovsxdq,           INS_pmovzxdq,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(AVX512F,         Divide,                                     64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_divps,              INS_divpd},             HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         DivideScalar,                               16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_divss,              INS_divsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         DuplicateEvenIndexed,                       64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movsldup,           INS_movddup},           HW_Category_SimpleSIMD,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX512F,         DuplicateOddIndexed,                        64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_movshdup,           INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX512F,         ExtractVector128,                           64,              2,     {INS_vextracti128,      INS_vextracti128,       INS_vextracti128,       INS_vextracti128,       INS_vextracti128,       INS_vextracti128,       INS_vextracti128,       INS_vextracti128,       INS_vextractf128,       INS_vextractf128},      HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ExtractVector256,                           64,              2,     {INS_vextracti32x8,     INS_vextracti32x8,      INS_vextracti32x8,      INS_vextracti32x8,      INS_vextracti32x8,      INS_vextracti32x8,      INS_vextracti64x4,      INS_vextracti64x4,      INS_vextractf64x4,      INS_vextractf64x4},     HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible|HW_Flag_NormalizeSmallTypeToInt)
HARDWARE_INTRINSIC(AVX512F,         Fixup,                                      64,              4,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfixupimmps,        INS_vfixupimmpd},       HW_Category_IMM,                    HW_Flag_SpecialImport|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         FixupScalar,                                16,              4,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfixupimmss,        INS_vfixupimmsd},       HW_Category_IMM,                    HW_Flag_SpecialImport|HW_Flag_FullRangeIMM|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX512F,         FusedMultiplyAdd,                           64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfmadd213ps,        INS_vfmadd213pd},       HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_EmbRoundingCompatible|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         FusedMultiplyAddNegated,                    64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfnmadd213ps,       INS_vfnmadd213pd},      HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_EmbRoundingCompatible|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         FusedMultiplyAddNegatedScalar,              16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfnmadd213ss,       INS_vfnmadd213sd},      HW_Category_SIMDScalar,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_EmbRoundingCompatible|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX512F,         FusedMultiplyAddScalar,                     16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfmadd213ss,        INS_vfmadd213sd},       HW_Category_SIMDScalar,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_EmbRoundingCompatible|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX512F,         FusedMultiplyAddSubtract,                   64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfmaddsub213ps,     INS_vfmaddsub213pd},    HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_EmbRoundingCompatible|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         FusedMultiplySubtract,                      64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfmsub213ps,        INS_vfmsub213pd},       HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_EmbRoundingCompatible|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         FusedMultiplySubtractAdd,                   64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfmsubadd213ps,     INS_vfmsubadd213pd},    HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_EmbRoundingCompatible|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         FusedMultiplySubtractNegated,               64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfnmsub213ps,       INS_vfnmsub213pd},      HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_EmbRoundingCompatible|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         FusedMultiplySubtractNegatedScalar,         16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfnmsub213ss,       INS_vfnmsub213sd},      HW_Category_SIMDScalar,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_EmbRoundingCompatible|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX512F,         FusedMultiplySubtractScalar,                16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfmsub213ss,        INS_vfmsub213sd},       HW_Category_SIMDScalar,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_EmbRoundingCompatible|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX512F,         GetExponent,                                64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vgetexpps,          INS_vgetexppd},         HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         GetExponentScalar,                          16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vgetexpss,          INS_vgetexpsd},         HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX512F,         GetMantissa,                                64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vgetmantps,         INS_vgetmantpd},        HW_Category_IMM,                    HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         GetMantissaScalar,                          16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vgetmantss,         INS_vgetmantsd},        HW_Category_IMM,                    HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX512F,         InsertVector128,                            64,              3,     {INS_vinserti128,       INS_vinserti128,        INS_vinserti128,        INS_vinserti128,        INS_vinserti128,        INS_vinserti128,        INS_vinserti128,        INS_vinserti128,        INS_vinsertf128,        INS_vinsertf128},       HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         InsertVector256,                            64,              3,     {INS_vinserti32x8,      INS_vinserti32x8,       INS_vinserti32x8,       INS_vinserti32x8,       INS_vinserti32x8,       INS_vinserti32x8,       INS_vinserti64x4,       INS_vinserti64x4,       INS_vinsertf64x4,       INS_vinsertf64x4},      HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible|HW_Flag_NormalizeSmallTypeToInt)
HARDWARE_INTRINSIC(AVX512F,         LoadAlignedVector512,                       64,              1,     {INS_movdqa,            INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_vmovdqa64,          INS_vmovdqa64,          INS_movaps,             INS_movapd},            HW_Category_MemoryLoad,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX512F,         LoadAlignedVector512NonTemporal,            64,              1,     {INS_movntdqa,          INS_movntdqa,           INS_movntdqa,           INS_movntdqa,           INS_movntdqa,           INS_movntdqa,           INS_movntdqa,           INS_movntdqa,           INS_invalid,            INS_invalid},           HW_Category_MemoryLoad,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX512F,         LoadVector512,                              64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512F,         Max,                                        64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_pmaxsd,             INS_pmaxud,             INS_vpmaxsq,            INS_vpmaxuq,            INS_maxps,              INS_maxpd},             HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_MaybeCommutative)
HARDWARE_INTRINSIC(AVX512F,         Min,                                        64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_pminsd,             INS_pminud,             INS_vpminsq,            INS_vpminuq,            INS_minps,              INS_minpd},             HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_MaybeCommutative)
HARDWARE_INTRINSIC(AVX512F,         Multiply,                                   64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_pmuldq,             INS_pmuludq,            INS_mulps,              INS_mulpd},             HW_Category_SimpleSIMD,             HW_Flag_MaybeCommutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         MultiplyLow,                                64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_pmulld,             INS_pmulld,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         MultiplyScalar,                             16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_mulss,              INS_mulsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         Or,                                         64,              2,     {INS_por,               INS_por,                INS_por,                INS_por,                INS_por,                INS_por,                INS_vporq,              INS_vporq,              INS_orps,               INS_orpd},              HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_NormalizeSmallTypeToInt)
HARDWARE_INTRINSIC(AVX512F,         Permute2x64,                                64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermilpd},         HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         Permute4x32,                                64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermilps,          INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         Permute4x64,                                64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermq,             INS_vpermq,             INS_invalid,            INS_vpermpd},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         PermuteVar16x32,                            64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermd,             INS_vpermd,             INS_invalid,            INS_invalid,            INS_vpermps,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         PermuteVar16x32x2,                          64,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermt2d,           INS_vpermt2d,           INS_invalid,            INS_invalid,            INS_vpermt2ps,          INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         PermuteVar2x64,                             64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermilpdvar},      HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         PermuteVar4x32,                             64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermilpsvar,       INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         PermuteVar8x64,                             64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermq_reg,         INS_vpermq_reg,         INS_invalid,            INS_vpermpd_reg},       HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         PermuteVar8x64x2,                           64,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermt2q,           INS_vpermt2q,           INS_invalid,            INS_vpermt2pd},         HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         Reciprocal14,                               64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrcp14ps,           INS_vrcp14pd},          HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         Reciprocal14Scalar,                         16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrcp14ss,           INS_vrcp14sd},          HW_Category_SimpleSIMD,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX512F,         ReciprocalSqrt14,                           64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrsqrt14ps,         INS_vrsqrt14pd},        HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ReciprocalSqrt14Scalar,                     16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrsqrt14ss,         INS_vrsqrt14sd},        HW_Category_SimpleSIMD,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX512F,         RotateLeft,                                 64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vprold,             INS_vprold,             INS_vprolq,             INS_vprolq,             INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_MaybeIMM|HW_Flag_MaybeNoJmpTableIMM|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         RotateLeftVariable,                         64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vprolvd,            INS_vprolvd,            INS_vprolvq,            INS_vprolvq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         RotateRight,                                64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vprord,             INS_vprord,             INS_vprorq,             INS_vprorq,             INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_MaybeIMM|HW_Flag_MaybeNoJmpTableIMM|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         RotateRightVariable,                        64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vprorvd,            INS_vprorvd,            INS_vprorvq,            INS_vprorvq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         RoundScale,                                 64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrndscaleps,        INS_vrndscalepd},       HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         RoundScaleScalar,                           16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrndscaless,        INS_vrndscalesd},       HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX512F,         Scale,                                      64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vscalefps,          INS_vscalefpd},         HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ScaleScalar,                                16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vscalefss,          INS_vscalefsd},         HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ShiftLeftLogical,                           64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_pslld,              INS_pslld,              INS_psllq,              INS_psllq,              INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_MaybeIMM|HW_Flag_NoJmpTableIMM|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ShiftLeftLogicalVariable,                   64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpsllvd,            INS_vpsllvd,            INS_vpsllvq,            INS_vpsllvq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ShiftRightArithmetic,                       64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_psrad,              INS_invalid,            INS_vpsraq,             INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_MaybeIMM|HW_Flag_NoJmpTableIMM|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ShiftRightArithmeticVariable,               64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpsravd,            INS_invalid,            INS_vpsravq,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ShiftRightLogical,                          64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_psrld,              INS_psrld,              INS_psrlq,              INS_psrlq,              INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_MaybeIMM|HW_Flag_NoJmpTableIMM|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         ShiftRightLogicalVariable,                  64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpsrlvd,            INS_vpsrlvd,            INS_vpsrlvq,            INS_vpsrlvq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         Shuffle,                                    64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_pshufd,             INS_pshufd,             INS_invalid,            INS_invalid,            INS_shufps,             INS_shufpd},            HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         Shuffle4x128,                               64,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vshufi32x4,         INS_vshufi32x4,         INS_vshufi64x2,         INS_vshufi64x2,         INS_vshuff32x4,         INS_vshuff64x2},        HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         Sqrt,                                       64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_sqrtps,             INS_sqrtpd},            HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         SqrtScalar,                                 16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_sqrtss,             INS_sqrtsd},            HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         Store,                                      64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(AVX512F,         StoreAligned,                               64,              2,     {INS_movdqa,            INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_movdqa,             INS_vmovdqa64,          INS_vmovdqa64,          INS_movaps,             INS_movapd},            HW_Category_MemoryStore,            HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(AVX512F,         StoreAlignedNonTemporal,                    64,              2,     {INS_movntdq,           INS_movntdq,            INS_movntdq,            INS_movntdq,            INS_movntdq,            INS_movntdq,            INS_movntdq,            INS_movntdq,            INS_movntps,            INS_movntpd},           HW_Category_MemoryStore,            HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(AVX512F,         Subtract,                                   64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_psubd,              INS_psubd,              INS_psubq,              INS_psubq,              INS_subps,              INS_subpd},             HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         SubtractScalar,                             16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_subss,              INS_subsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F,         TernaryLogic,                               64,              4,     {INS_vpternlogd,        INS_vpternlogd,         INS_vpternlogd,         INS_vpternlogd,         INS_vpternlogd,         INS_vpternlogd,         INS_vpternlogq,         INS_vpternlogq,         INS_vpternlogd,         INS_vpternlogq},        HW_Category_IMM,                    HW_Flag_SpecialImport|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_NormalizeSmallTypeToInt)
HARDWARE_INTRINSIC(AVX512F,         UnpackHigh,                                 64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_punpckhdq,          INS_punpckhdq,          INS_punpckhqdq,         INS_punpckhqdq,         INS_unpckhps,           INS_unpckhpd},          HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         UnpackLow,                                  64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_punpckldq,          INS_punpckldq,          INS_punpcklqdq,         INS_punpcklqdq,         INS_unpcklps,           INS_unpcklpd},          HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F,         Xor,                                        64,              2,     {INS_pxor,              INS_pxor,               INS_pxor,               INS_pxor,               INS_pxor,               INS_pxor,               INS_vpxorq,             INS_vpxorq,             INS_xorps,              INS_xorpd},             HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_CanBenefitFromConstantProp|HW_Flag_NormalizeSmallTypeToInt)
#define LAST_NI_AVX512F             NI_AVX512F_Xor

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  AVX512F.VL Intrinsics
#define FIRST_NI_AVX512F_VL         NI_AVX512F_VL_Abs
HARDWARE_INTRINSIC(AVX512F_VL,      Abs,                                        -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpabsq,             INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      AlignRight32,                               -1,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_valignd,            INS_valignd,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      AlignRight64,                               -1,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_valignq,            INS_valignq,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      CompareGreaterThan,                         -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512F_VL,      CompareGreaterThanOrEqual,                  -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512F_VL,      CompareLessThan,                            -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512F_VL,      CompareLessThanOrEqual,                     -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512F_VL,      CompareNotEqual,                            -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512F_VL,      ConvertToVector128Byte,                     -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovdb,            INS_vpmovdb,            INS_vpmovqb,            INS_vpmovqb,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F_VL,      ConvertToVector128ByteWithSaturation,       -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovusdb,          INS_invalid,            INS_vpmovusqb,          INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F_VL,      ConvertToVector128Double,                   16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtudq2pd,         INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      ConvertToVector128Int16,                    -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovdw,            INS_vpmovdw,            INS_vpmovqw,            INS_vpmovqw,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F_VL,      ConvertToVector128Int16WithSaturation,      -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovsdw,           INS_invalid,            INS_vpmovsqw,           INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F_VL,      ConvertToVector128Int32,                    -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovqd,            INS_vpmovqd,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F_VL,      ConvertToVector128Int32WithSaturation,      -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovsqd,           INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F_VL,      ConvertToVector128SByte,                    -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovdb,            INS_vpmovdb,            INS_vpmovqb,            INS_vpmovqb,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F_VL,      ConvertToVector128SByteWithSaturation,      -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovsdb,           INS_invalid,            INS_vpmovsqb,           INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F_VL,      ConvertToVector128Single,                   16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtudq2ps,         INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      ConvertToVector128UInt16,                   -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovdw,            INS_vpmovdw,            INS_vpmovqw,            INS_vpmovqw,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F_VL,      ConvertToVector128UInt16WithSaturation,     -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovusdw,          INS_invalid,            INS_vpmovusqw,          INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F_VL,      ConvertToVector128UInt32,                   -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovqd,            INS_vpmovqd,            INS_vcvtps2udq,         INS_vcvtpd2udq},        HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F_VL,      ConvertToVector128UInt32WithSaturation,     -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovusqd,          INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512F_VL,      ConvertToVector128UInt32WithTruncation,     -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttps2udq,        INS_vcvttpd2udq},       HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      ConvertToVector256Double,                   32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtudq2pd,         INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      ConvertToVector256Single,                   32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtudq2ps,         INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      ConvertToVector256UInt32,                   32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtps2udq,         INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      ConvertToVector256UInt32WithTruncation,     32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttps2udq,        INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      Fixup,                                      -1,              4,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfixupimmps,        INS_vfixupimmpd},       HW_Category_IMM,                    HW_Flag_SpecialImport|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      GetExponent,                                -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vgetexpps,          INS_vgetexppd},         HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      GetMantissa,                                -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vgetmantps,         INS_vgetmantpd},        HW_Category_IMM,                    HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      Max,                                        -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmaxsq,            INS_vpmaxuq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX512F_VL,      Min,                                        -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpminsq,            INS_vpminuq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX512F_VL,      PermuteVar2x64x2,                           16,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermt2q,           INS_vpermt2q,           INS_invalid,            INS_vpermt2pd},         HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      PermuteVar4x32x2,                           16,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermt2d,           INS_vpermt2d,           INS_invalid,            INS_invalid,            INS_vpermt2ps,          INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      PermuteVar4x64,                             32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermq_reg,         INS_vpermq_reg,         INS_invalid,            INS_vpermpd_reg},       HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      PermuteVar4x64x2,                           32,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermt2q,           INS_vpermt2q,           INS_invalid,            INS_vpermt2pd},         HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      PermuteVar8x32x2,                           32,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermt2d,           INS_vpermt2d,           INS_invalid,            INS_invalid,            INS_vpermt2ps,          INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      Reciprocal14,                               -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrcp14ps,           INS_vrcp14pd},          HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      ReciprocalSqrt14,                           -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrsqrt14ps,         INS_vrsqrt14pd},        HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      RotateLeft,                                 -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vprold,             INS_vprold,             INS_vprolq,             INS_vprolq,             INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_MaybeIMM|HW_Flag_MaybeNoJmpTableIMM|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      RotateLeftVariable,                         -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vprolvd,            INS_vprolvd,            INS_vprolvq,            INS_vprolvq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      RotateRight,                                -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vprord,             INS_vprord,             INS_vprorq,             INS_vprorq,             INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_MaybeIMM|HW_Flag_MaybeNoJmpTableIMM|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      RotateRightVariable,                        -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vprorvd,            INS_vprorvd,            INS_vprorvq,            INS_vprorvq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      RoundScale,                                 -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrndscaleps,        INS_vrndscalepd},       HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      Scale,                                      -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vscalefps,          INS_vscalefpd},         HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      ShiftRightArithmetic,                       -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpsraq,             INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_MaybeIMM|HW_Flag_NoJmpTableIMM|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      ShiftRightArithmeticVariable,               -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpsravq,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      Shuffle2x128,                               32,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vshufi32x4,         INS_vshufi32x4,         INS_vshufi64x2,         INS_vshufi64x2,         INS_vshuff32x4,         INS_vshuff64x2},        HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512F_VL,      TernaryLogic,                               -1,              4,     {INS_vpternlogd,        INS_vpternlogd,         INS_vpternlogd,         INS_vpternlogd,         INS_vpternlogd,         INS_vpternlogd,         INS_vpternlogq,         INS_vpternlogq,         INS_vpternlogd,         INS_vpternlogq},        HW_Category_IMM,                    HW_Flag_SpecialImport|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_NormalizeSmallTypeToInt)
#define LAST_NI_AVX512F_VL          NI_AVX512F_VL_TernaryLogic

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  AVX512F.X64 Intrinsics
#define FIRST_NI_AVX512F_X64        NI_AVX512F_X64_ConvertScalarToVector128Double
HARDWARE_INTRINSIC(AVX512F_X64,     ConvertScalarToVector128Double,             16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtsi2sd64,         INS_vcvtusi2sd64,       INS_invalid,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromSecondArg|HW_Flag_CopyUpperBits|HW_Flag_SpecialCodeGen|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F_X64,     ConvertScalarToVector128Single,             16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtsi2ss64,         INS_vcvtusi2ss64,       INS_invalid,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromSecondArg|HW_Flag_CopyUpperBits|HW_Flag_SpecialCodeGen|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F_X64,     ConvertToInt64,                             16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtss2si,           INS_cvtsd2si},          HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F_X64,     ConvertToUInt64,                            16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtss2usi,         INS_vcvtsd2usi},        HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512F_X64,     ConvertToUInt64WithTruncation,              16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttss2usi64,      INS_vcvttsd2usi64},     HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
#define LAST_NI_AVX512F_X64         NI_AVX512F_X64_ConvertToUInt64WithTruncation

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  AVX512BW Intrinsics
#define FIRST_NI_AVX512BW            NI_AVX512BW_Abs
HARDWARE_INTRINSIC(AVX512BW,        Abs,                                        64,              1,     {INS_pabsb,             INS_invalid,            INS_pabsw,              INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        Add,                                        64,              2,     {INS_paddb,             INS_paddb,              INS_paddw,              INS_paddw,              INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        AddSaturate,                                64,              2,     {INS_paddsb,            INS_paddusb,            INS_paddsw,             INS_paddusw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        AlignRight,                                 64,              3,     {INS_palignr,           INS_palignr,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        Average,                                    64,              2,     {INS_invalid,           INS_pavgb,              INS_invalid,            INS_pavgw,              INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        BlendVariable,                              64,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512BW,        BroadcastScalarToVector512,                 64,              1,     {INS_vpbroadcastb,      INS_vpbroadcastb,       INS_vpbroadcastw,       INS_vpbroadcastw,       INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_MaybeMemoryLoad)
HARDWARE_INTRINSIC(AVX512BW,        CompareEqual,                               64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512BW,        CompareGreaterThan,                         64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512BW,        CompareGreaterThanOrEqual,                  64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512BW,        CompareLessThan,                            64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512BW,        CompareLessThanOrEqual,                     64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512BW,        CompareNotEqual,                            64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512BW,        ConvertToVector256Byte,                     64,              1,     {INS_invalid,           INS_invalid,            INS_vpmovwb,            INS_vpmovwb,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512BW,        ConvertToVector256ByteWithSaturation,       64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_vpmovuswb,          INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512BW,        ConvertToVector256SByte,                    64,              1,     {INS_invalid,           INS_invalid,            INS_vpmovwb,            INS_vpmovwb,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512BW,        ConvertToVector256SByteWithSaturation,      64,              1,     {INS_invalid,           INS_invalid,            INS_vpmovswb,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512BW,        ConvertToVector512Int16,                    64,              1,     {INS_pmovsxbw,          INS_pmovzxbw,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(AVX512BW,        ConvertToVector512UInt16,                   64,              1,     {INS_pmovsxbw,          INS_pmovzxbw,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg)
HARDWARE_INTRINSIC(AVX512BW,        LoadVector512,                              64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512BW,        Max,                                        64,              2,     {INS_pmaxsb,            INS_pmaxub,             INS_pmaxsw,             INS_pmaxuw,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        Min,                                        64,              2,     {INS_pminsb,            INS_pminub,             INS_pminsw,             INS_pminuw,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        MultiplyAddAdjacent,                        64,              2,     {INS_invalid,           INS_invalid,            INS_pmaddubsw,          INS_invalid,            INS_pmaddwd,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        MultiplyHigh,                               64,              2,     {INS_invalid,           INS_invalid,            INS_pmulhw,             INS_pmulhuw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        MultiplyHighRoundScale,                     64,              2,     {INS_invalid,           INS_invalid,            INS_pmulhrsw,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        MultiplyLow,                                64,              2,     {INS_invalid,           INS_invalid,            INS_pmullw,             INS_pmullw,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        PackSignedSaturate,                         64,              2,     {INS_packsswb,          INS_invalid,            INS_packssdw,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        PackUnsignedSaturate,                       64,              2,     {INS_invalid,           INS_packuswb,           INS_invalid,            INS_packusdw,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        PermuteVar32x16,                            64,              2,     {INS_invalid,           INS_invalid,            INS_vpermw,             INS_vpermw,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        PermuteVar32x16x2,                          64,              3,     {INS_invalid,           INS_invalid,            INS_vpermt2w,           INS_vpermt2w,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        ShiftLeftLogical,                           64,              2,     {INS_invalid,           INS_invalid,            INS_psllw,              INS_psllw,              INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_MaybeIMM|HW_Flag_NoJmpTableIMM|HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        ShiftLeftLogical128BitLane,                 64,              2,     {INS_pslldq,            INS_pslldq,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(AVX512BW,        ShiftLeftLogicalVariable,                   64,              2,     {INS_invalid,           INS_invalid,            INS_vpsllvw,            INS_vpsllvw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        ShiftRightArithmetic,                       64,              2,     {INS_invalid,           INS_invalid,            INS_psraw,              INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_MaybeIMM|HW_Flag_NoJmpTableIMM|HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        ShiftRightArithmeticVariable,               64,              2,     {INS_invalid,           INS_invalid,            INS_vpsravw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        ShiftRightLogical,                          64,              2,     {INS_invalid,           INS_invalid,            INS_psrlw,              INS_psrlw,              INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_MaybeIMM|HW_Flag_NoJmpTableIMM|HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        ShiftRightLogical128BitLane,                64,              2,     {INS_psrldq,            INS_psrldq,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM)
HARDWARE_INTRINSIC(AVX512BW,        ShiftRightLogicalVariable,                  64,              2,     {INS_invalid,           INS_invalid,            INS_vpsrlvw,            INS_vpsrlvw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        Shuffle,                                    64,              2,     {INS_pshufb,            INS_pshufb,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        ShuffleHigh,                                64,              2,     {INS_invalid,           INS_invalid,            INS_pshufhw,            INS_pshufhw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        ShuffleLow,                                 64,              2,     {INS_invalid,           INS_invalid,            INS_pshuflw,            INS_pshuflw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        Store,                                      64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Helper,                 HW_Flag_InvalidNodeId|HW_Flag_BaseTypeFromSecondArg)
HARDWARE_INTRINSIC(AVX512BW,        Subtract,                                   64,              2,     {INS_psubb,             INS_psubb,              INS_psubw,              INS_psubw,              INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        SubtractSaturate,                           64,              2,     {INS_psubsb,            INS_psubusb,            INS_psubsw,             INS_psubusw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        SumAbsoluteDifferences,                     64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_psadbw,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX512BW,        SumAbsoluteDifferencesInBlock32,            64,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_vdbpsadbw,          INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        UnpackHigh,                                 64,              2,     {INS_punpckhbw,         INS_punpckhbw,          INS_punpckhwd,          INS_punpckhwd,          INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW,        UnpackLow,                                  64,              2,     {INS_punpcklbw,         INS_punpcklbw,          INS_punpcklwd,          INS_punpcklwd,          INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
#define LAST_NI_AVX512BW            NI_AVX512BW_UnpackLow

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  AVX512BW.VL Intrinsics
#define FIRST_NI_AVX512BW_VL        NI_AVX512BW_VL_CompareGreaterThan
HARDWARE_INTRINSIC(AVX512BW_VL,     CompareGreaterThan,                         -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512BW_VL,     CompareGreaterThanOrEqual,                  -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512BW_VL,     CompareLessThan,                            -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512BW_VL,     CompareLessThanOrEqual,                     -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512BW_VL,     CompareNotEqual,                            -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX512BW_VL,     ConvertToVector128Byte,                     -1,              1,     {INS_invalid,           INS_invalid,            INS_vpmovwb,            INS_vpmovwb,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512BW_VL,     ConvertToVector128ByteWithSaturation,       -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_vpmovuswb,          INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512BW_VL,     ConvertToVector128SByte,                    -1,              1,     {INS_invalid,           INS_invalid,            INS_vpmovwb,            INS_vpmovwb,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512BW_VL,     ConvertToVector128SByteWithSaturation,      -1,              1,     {INS_invalid,           INS_invalid,            INS_vpmovswb,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX512BW_VL,     PermuteVar16x16,                            32,              2,     {INS_invalid,           INS_invalid,            INS_vpermw,             INS_vpermw,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW_VL,     PermuteVar16x16x2,                          32,              3,     {INS_invalid,           INS_invalid,            INS_vpermt2w,           INS_vpermt2w,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW_VL,     PermuteVar8x16 ,                            16,              2,     {INS_invalid,           INS_invalid,            INS_vpermw,             INS_vpermw,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW_VL,     PermuteVar8x16x2,                           16,              3,     {INS_invalid,           INS_invalid,            INS_vpermt2w,           INS_vpermt2w,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW_VL,     ShiftLeftLogicalVariable,                   -1,              2,     {INS_invalid,           INS_invalid,            INS_vpsllvw,            INS_vpsllvw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW_VL,     ShiftRightArithmeticVariable,               -1,              2,     {INS_invalid,           INS_invalid,            INS_vpsravw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW_VL,     ShiftRightLogicalVariable,                  -1,              2,     {INS_invalid,           INS_invalid,            INS_vpsrlvw,            INS_vpsrlvw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512BW_VL,     SumAbsoluteDifferencesInBlock32,            -1,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_vdbpsadbw,          INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
#define LAST_NI_AVX512BW_VL         NI_AVX512BW_VL_SumAbsoluteDifferencesInBlock32

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  AVX512CD Intrinsics
#define FIRST_NI_AVX512CD           NI_AVX512CD_DetectConflicts
HARDWARE_INTRINSIC(AVX512CD,        DetectConflicts,                            64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpconflictd,        INS_vpconflictd,        INS_vpconflictq,        INS_vpconflictq,        INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512CD,        LeadingZeroCount,                           64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vplzcntd,           INS_vplzcntd,           INS_vplzcntq,           INS_vplzcntq,           INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
#define LAST_NI_AVX512CD            NI_AVX512CD_LeadingZeroCount

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  AVX512CD.VL Intrinsics
#define FIRST_NI_AVX512CD_VL        NI_AVX512CD_VL_DetectConflicts
HARDWARE_INTRINSIC(AVX512CD_VL,     DetectConflicts,                            -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpconflictd,        INS_vpconflictd,        INS_vpconflictq,        INS_vpconflictq,        INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512CD_VL,     LeadingZeroCount,                           -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vplzcntd,           INS_vplzcntd,           INS_vplzcntq,           INS_vplzcntq,           INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
#define LAST_NI_AVX512CD_VL         NI_AVX512CD_VL_LeadingZeroCount

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  AVX512DQ Intrinsics
#define FIRST_NI_AVX512DQ           NI_AVX512DQ_And
HARDWARE_INTRINSIC(AVX512DQ,        And,                                        64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_andps,              INS_andpd},             HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ,        AndNot,                                     64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_andnps,             INS_andnpd},            HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ,        BroadcastPairScalarToVector512,             64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vbroadcasti32x2,    INS_vbroadcasti32x2,    INS_invalid,            INS_invalid,            INS_vbroadcastf32x2,    INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX512DQ,        BroadcastVector128ToVector512,              64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vbroadcasti64x2,    INS_vbroadcasti64x2,    INS_invalid,            INS_vbroadcastf64x2},   HW_Category_MemoryLoad,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX512DQ,        BroadcastVector256ToVector512,              64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vbroadcasti32x8,    INS_vbroadcasti32x8,    INS_invalid,            INS_invalid,            INS_vbroadcastf32x8,    INS_invalid},           HW_Category_MemoryLoad,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX512DQ,        ConvertToVector256Single,                   64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtqq2ps,          INS_vcvtuqq2ps,         INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512DQ,        ConvertToVector512Double,                   64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtqq2pd,          INS_vcvtuqq2pd,         INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512DQ,        ConvertToVector512Int64,                    64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtps2qq,          INS_vcvtpd2qq},         HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512DQ,        ConvertToVector512Int64WithTruncation,      64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttps2qq,         INS_vcvttpd2qq},        HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ,        ConvertToVector512UInt64,                   64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtps2uqq,         INS_vcvtpd2uqq},        HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX512DQ,        ConvertToVector512UInt64WithTruncation,     64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttps2uqq,        INS_vcvttpd2uqq},       HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ,        ExtractVector128,                           64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vextracti64x2,      INS_vextracti64x2,      INS_invalid,            INS_vextractf64x2},     HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ,        ExtractVector256,                           64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vextracti32x8,      INS_vextracti32x8,      INS_invalid,            INS_invalid,            INS_vextractf32x8,      INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ,        InsertVector128,                            64,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vinserti64x2,       INS_vinserti64x2,       INS_invalid,            INS_vinsertf64x2},      HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ,        InsertVector256,                            64,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vinserti32x8,       INS_vinserti32x8,       INS_invalid,            INS_invalid,            INS_vinsertf32x8,       INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ,        MultiplyLow,                                64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmullq,            INS_vpmullq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ,        Or,                                         64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_orps,               INS_orpd},              HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ,        Range,                                      64,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrangeps,           INS_vrangepd},          HW_Category_IMM,                    HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ,        RangeScalar,                                16,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrangess,           INS_vrangesd},          HW_Category_IMM,                    HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX512DQ,        Reduce,                                     64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vreduceps,          INS_vreducepd},         HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ,        ReduceScalar,                               16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vreducess,          INS_vreducesd},         HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX512DQ,        Xor,                                        64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_xorps,              INS_xorpd},             HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_CanBenefitFromConstantProp)
#define LAST_NI_AVX512DQ            NI_AVX512DQ_Xor

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  AVX512DQ.VL Intrinsics
#define FIRST_NI_AVX512DQ_VL        NI_AVX512DQ_VL_BroadcastPairScalarToVector128
HARDWARE_INTRINSIC(AVX512DQ_VL,     BroadcastPairScalarToVector128,             16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vbroadcasti32x2,    INS_vbroadcasti32x2,    INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX512DQ_VL,     BroadcastPairScalarToVector256,             32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vbroadcasti32x2,    INS_vbroadcasti32x2,    INS_invalid,            INS_invalid,            INS_vbroadcastf32x2,    INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX512DQ_VL,     ConvertToVector128Double,                   16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtqq2pd,          INS_vcvtuqq2pd,         INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ_VL,     ConvertToVector128Int64,                    16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtps2qq,          INS_vcvtpd2qq},         HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ_VL,     ConvertToVector128Int64WithTruncation,      16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttps2qq,         INS_vcvttpd2qq},        HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ_VL,     ConvertToVector128Single,                   -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtqq2ps,          INS_vcvtuqq2ps,         INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ_VL,     ConvertToVector128UInt64,                   16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtps2uqq,         INS_vcvtpd2uqq},        HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ_VL,     ConvertToVector128UInt64WithTruncation,     16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttps2uqq,        INS_vcvttpd2uqq},       HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ_VL,     ConvertToVector256Double,                   32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtqq2pd,          INS_vcvtuqq2pd,         INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ_VL,     ConvertToVector256Int64,                    32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtps2qq,          INS_vcvtpd2qq},         HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ_VL,     ConvertToVector256Int64WithTruncation,      32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttps2qq,         INS_vcvttpd2qq},        HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ_VL,     ConvertToVector256UInt64,                   32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtps2uqq,         INS_vcvtpd2uqq},        HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ_VL,     ConvertToVector256UInt64WithTruncation,     32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttps2uqq,        INS_vcvttpd2uqq},       HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ_VL,     MultiplyLow,                                -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmullq,            INS_vpmullq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ_VL,     Range,                                      -1,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrangeps,           INS_vrangepd},          HW_Category_IMM,                    HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512DQ_VL,     Reduce,                                     -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vreduceps,          INS_vreducepd},         HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
#define LAST_NI_AVX512DQ_VL         NI_AVX512DQ_VL_Reduce

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  AVX512VBMI Intrinsics
#define FIRST_NI_AVX512VBMI         NI_AVX512VBMI_MultiShift
HARDWARE_INTRINSIC(AVX512VBMI,      MultiShift,                                 64,              2,     {INS_vpmultishiftqb,    INS_vpmultishiftqb,     INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512VBMI,      PermuteVar64x8,                             64,              2,     {INS_vpermb,            INS_vpermb,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512VBMI,      PermuteVar64x8x2,                           64,              3,     {INS_vpermt2b,          INS_vpermt2b,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbMaskingCompatible)
#define LAST_NI_AVX512VBMI          NI_AVX512VBMI_PermuteVar64x8x2

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  AVX512VBMI.VL Intrinsics
#define FIRST_NI_AVX512VBMI_VL      NI_AVX512VBMI_VL_MultiShift
HARDWARE_INTRINSIC(AVX512VBMI_VL,   MultiShift,                                 -1,              2,     {INS_vpmultishiftqb,    INS_vpmultishiftqb,     INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512VBMI_VL,   PermuteVar16x8,                             16,              2,     {INS_vpermb,            INS_vpermb,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512VBMI_VL,   PermuteVar16x8x2,                           16,              3,     {INS_vpermt2b,          INS_vpermt2b,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512VBMI_VL,   PermuteVar32x8,                             32,              2,     {INS_vpermb,            INS_vpermb,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX512VBMI_VL,   PermuteVar32x8x2,                           32,              3,     {INS_vpermt2b,          INS_vpermt2b,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbMaskingCompatible)
#define LAST_NI_AVX512VBMI_VL       NI_AVX512VBMI_VL_PermuteVar32x8x2

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  AVX10V1 Intrinsics
#define FIRST_NI_AVX10v1            NI_AVX10v1_Abs
HARDWARE_INTRINSIC(AVX10v1,         Abs,                                        -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpabsq,             INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         AddScalar,                                  16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_addss,              INS_addsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         AlignRight32,                               -1,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_valignd,            INS_valignd,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         AlignRight64,                               -1,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_valignq,            INS_valignq,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         BroadcastPairScalarToVector128,             16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vbroadcasti32x2,    INS_vbroadcasti32x2,    INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX10v1,         BroadcastPairScalarToVector256,             32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vbroadcasti32x2,    INS_vbroadcasti32x2,    INS_invalid,            INS_invalid,            INS_vbroadcastf32x2,    INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX10v1,         CompareGreaterThan,                         -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX10v1,         CompareGreaterThanOrEqual,                  -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX10v1,         CompareLessThan,                            -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX10v1,         CompareLessThanOrEqual,                     -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX10v1,         CompareNotEqual,                            -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_InvalidNodeId)
HARDWARE_INTRINSIC(AVX10v1,         ConvertScalarToVector128Double,             16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtsi2sd32,         INS_vcvtusi2sd32,       INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromSecondArg|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX10v1,         ConvertScalarToVector128Single,             16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtsi2ss32,         INS_vcvtusi2ss32,       INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtsd2ss},          HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromSecondArg|HW_Flag_CopyUpperBits|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToInt32,                             16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtss2si,           INS_cvtsd2si},          HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToUInt32,                            16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtss2usi,         INS_vcvtsd2usi},        HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToUInt32WithTruncation,              16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttss2usi32,      INS_vcvttsd2usi32},     HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector128Byte,                     -1,              1,     {INS_invalid,           INS_invalid,            INS_vpmovwb,            INS_vpmovwb,            INS_vpmovdb,            INS_vpmovdb,            INS_vpmovqb,            INS_vpmovqb,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector128ByteWithSaturation,       -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_vpmovuswb,          INS_invalid,            INS_vpmovusdb,          INS_invalid,            INS_vpmovusqb,          INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector128Double,                   16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtudq2pd,         INS_vcvtqq2pd,          INS_vcvtuqq2pd,         INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector128Int16,                    -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovdw,            INS_vpmovdw,            INS_vpmovqw,            INS_vpmovqw,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector128Int16WithSaturation,      -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovsdw,           INS_invalid,            INS_vpmovsqw,           INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector128Int32,                    -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovqd,            INS_vpmovqd,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector128Int32WithSaturation,      -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovsqd,           INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector128Int64,                    16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtps2qq,          INS_vcvtpd2qq},         HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector128Int64WithTruncation,      16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttps2qq,         INS_vcvttpd2qq},        HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector128SByte,                    -1,              1,     {INS_invalid,           INS_invalid,            INS_vpmovwb,            INS_vpmovwb,            INS_vpmovdb,            INS_vpmovdb,            INS_vpmovqb,            INS_vpmovqb,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector128SByteWithSaturation,      -1,              1,     {INS_invalid,           INS_invalid,            INS_vpmovswb,           INS_invalid,            INS_vpmovsdb,           INS_invalid,            INS_vpmovsqb,           INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector128Single,                   -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtudq2ps,         INS_vcvtqq2ps,          INS_vcvtuqq2ps,         INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector128UInt16,                   -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovdw,            INS_vpmovdw,            INS_vpmovqw,            INS_vpmovqw,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector128UInt16WithSaturation,     -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovusdw,          INS_invalid,            INS_vpmovusqw,          INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector128UInt32,                   -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovqd,            INS_vpmovqd,            INS_vcvtps2udq,         INS_vcvtpd2udq},        HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector128UInt32WithSaturation,     -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmovusqd,          INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector128UInt32WithTruncation,     -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttps2udq,        INS_vcvttpd2udq},       HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector128UInt64,                   16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtps2uqq,         INS_vcvtpd2uqq},        HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector128UInt64WithTruncation,     16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttps2uqq,        INS_vcvttpd2uqq},       HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector256Double,                   32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtudq2pd,         INS_vcvtqq2pd,          INS_vcvtuqq2pd,         INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector256Int64,                    32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtps2qq,          INS_vcvtpd2qq},         HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector256Int64WithTruncation,      32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttps2qq,         INS_vcvttpd2qq},        HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector256Single,                   32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtudq2ps,         INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector256UInt32,                   32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtps2udq,         INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector256UInt32WithTruncation,     32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttps2udq,        INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector256UInt64,                   32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtps2uqq,         INS_vcvtpd2uqq},        HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ConvertToVector256UInt64WithTruncation,     32,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttps2uqq,        INS_vcvttpd2uqq},       HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         DetectConflicts,                            -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpconflictd,        INS_vpconflictd,        INS_vpconflictq,        INS_vpconflictq,        INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         DivideScalar,                               16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_divss,              INS_divsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         Fixup,                                      -1,              4,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfixupimmps,        INS_vfixupimmpd},       HW_Category_IMM,                    HW_Flag_SpecialImport|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         FixupScalar,                                16,              4,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfixupimmss,        INS_vfixupimmsd},       HW_Category_IMM,                    HW_Flag_SpecialImport|HW_Flag_FullRangeIMM|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX10v1,         FusedMultiplyAddNegatedScalar,              16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfnmadd213ss,       INS_vfnmadd213sd},      HW_Category_SIMDScalar,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_EmbRoundingCompatible|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX10v1,         FusedMultiplyAddScalar,                     16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfmadd213ss,        INS_vfmadd213sd},       HW_Category_SIMDScalar,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_EmbRoundingCompatible|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX10v1,         FusedMultiplySubtractNegatedScalar,         16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfnmsub213ss,       INS_vfnmsub213sd},      HW_Category_SIMDScalar,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_EmbRoundingCompatible|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX10v1,         FusedMultiplySubtractScalar,                16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfmsub213ss,        INS_vfmsub213sd},       HW_Category_SIMDScalar,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_EmbRoundingCompatible|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX10v1,         GetExponent,                                -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vgetexpps,          INS_vgetexppd},         HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         GetExponentScalar,                          16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vgetexpss,          INS_vgetexpsd},         HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX10v1,         GetMantissa,                                -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vgetmantps,         INS_vgetmantpd},        HW_Category_IMM,                    HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         GetMantissaScalar,                          16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vgetmantss,         INS_vgetmantsd},        HW_Category_IMM,                    HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX10v1,         LeadingZeroCount,                           -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vplzcntd,           INS_vplzcntd,           INS_vplzcntq,           INS_vplzcntq,           INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         Max,                                        -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmaxsq,            INS_vpmaxuq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX10v1,         Min,                                        -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpminsq,            INS_vpminuq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_Commutative)
HARDWARE_INTRINSIC(AVX10v1,         MultiShift,                                 -1,              2,     {INS_vpmultishiftqb,    INS_vpmultishiftqb,     INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         MultiplyLow,                                -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmullq,            INS_vpmullq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         MultiplyScalar,                             16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_mulss,              INS_mulsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         PermuteVar16x16,                            32,              2,     {INS_invalid,           INS_invalid,            INS_vpermw,             INS_vpermw,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         PermuteVar16x16x2,                          32,              3,     {INS_invalid,           INS_invalid,            INS_vpermt2w,           INS_vpermt2w,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         PermuteVar16x8,                             16,              2,     {INS_vpermb,            INS_vpermb,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         PermuteVar16x8x2,                           16,              3,     {INS_vpermt2b,          INS_vpermt2b,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         PermuteVar2x64x2,                           16,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermt2q,           INS_vpermt2q,           INS_invalid,            INS_vpermt2pd},         HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         PermuteVar32x8,                             32,              2,     {INS_vpermb,            INS_vpermb,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         PermuteVar32x8x2,                           32,              3,     {INS_vpermt2b,          INS_vpermt2b,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         PermuteVar4x32x2,                           16,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermt2d,           INS_vpermt2d,           INS_invalid,            INS_invalid,            INS_vpermt2ps,          INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         PermuteVar4x64,                             32,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermq_reg,         INS_vpermq_reg,         INS_invalid,            INS_vpermpd_reg},       HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         PermuteVar4x64x2,                           32,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermt2q,           INS_vpermt2q,           INS_invalid,            INS_vpermt2pd},         HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         PermuteVar8x16,                             16,              2,     {INS_invalid,           INS_invalid,            INS_vpermw,             INS_vpermw,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         PermuteVar8x16x2,                           16,              3,     {INS_invalid,           INS_invalid,            INS_vpermt2w,           INS_vpermt2w,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         PermuteVar8x32x2,                           32,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpermt2d,           INS_vpermt2d,           INS_invalid,            INS_invalid,            INS_vpermt2ps,          INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         Range,                                      -1,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrangeps,           INS_vrangepd},          HW_Category_IMM,                    HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         RangeScalar,                                16,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrangess,           INS_vrangesd},          HW_Category_IMM,                    HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX10v1,         Reciprocal14,                               -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrcp14ps,           INS_vrcp14pd},          HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         Reciprocal14Scalar,                         16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrcp14ss,           INS_vrcp14sd},          HW_Category_SimpleSIMD,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX10v1,         ReciprocalSqrt14,                           -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrsqrt14ps,         INS_vrsqrt14pd},        HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ReciprocalSqrt14Scalar,                     16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrsqrt14ss,         INS_vrsqrt14sd},        HW_Category_SimpleSIMD,             HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX10v1,         Reduce,                                     -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vreduceps,          INS_vreducepd},         HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ReduceScalar,                               16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vreducess,          INS_vreducesd},         HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX10v1,         RotateLeft,                                 -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vprold,             INS_vprold,             INS_vprolq,             INS_vprolq,             INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_MaybeIMM|HW_Flag_MaybeNoJmpTableIMM|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         RotateLeftVariable,                         -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vprolvd,            INS_vprolvd,            INS_vprolvq,            INS_vprolvq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         RotateRight,                                -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vprord,             INS_vprord,             INS_vprorq,             INS_vprorq,             INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_MaybeIMM|HW_Flag_MaybeNoJmpTableIMM|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         RotateRightVariable,                        -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vprorvd,            INS_vprorvd,            INS_vprorvq,            INS_vprorvq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         RoundScale,                                 -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrndscaleps,        INS_vrndscalepd},       HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         RoundScaleScalar,                           16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrndscaless,        INS_vrndscalesd},       HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(AVX10v1,         Scale,                                      -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vscalefps,          INS_vscalefpd},         HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ScaleScalar,                                16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vscalefss,          INS_vscalefsd},         HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ShiftLeftLogicalVariable,                   -1,              2,     {INS_invalid,           INS_invalid,            INS_vpsllvw,            INS_vpsllvw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ShiftRightArithmetic,                       -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpsraq,             INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_MaybeIMM|HW_Flag_NoJmpTableIMM|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ShiftRightArithmeticVariable,               -1,              2,     {INS_invalid,           INS_invalid,            INS_vpsravw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpsravq,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible)
HARDWARE_INTRINSIC(AVX10v1,         ShiftRightLogicalVariable,                  -1,              2,     {INS_invalid,           INS_invalid,            INS_vpsrlvw,            INS_vpsrlvw,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         Shuffle2x128,                               32,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vshufi32x4,         INS_vshufi32x4,         INS_vshufi64x2,         INS_vshufi64x2,         INS_vshuff32x4,         INS_vshuff64x2},        HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         SqrtScalar,                                 16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_sqrtss,             INS_sqrtsd},            HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         SubtractScalar,                             16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_subss,              INS_subsd},             HW_Category_SIMDScalar,             HW_Flag_CopyUpperBits|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         SumAbsoluteDifferencesInBlock32,            -1,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_vdbpsadbw,          INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1,         TernaryLogic,                               -1,              4,     {INS_vpternlogd,        INS_vpternlogd,         INS_vpternlogd,         INS_vpternlogd,         INS_vpternlogd,         INS_vpternlogd,         INS_vpternlogq,         INS_vpternlogq,         INS_vpternlogd,         INS_vpternlogq},        HW_Category_IMM,                    HW_Flag_SpecialImport|HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_NormalizeSmallTypeToInt)
#define LAST_NI_AVX10v1             NI_AVX10v1_TernaryLogic

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA                  Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  AVX10V1_V512 Intrinsics
#define FIRST_NI_AVX10v1_V512           NI_AVX10v1_V512_And
HARDWARE_INTRINSIC(AVX10v1_V512,        And,                                        64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_andps,              INS_andpd},             HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        AndNot,                                     64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_andnps,             INS_andnpd},            HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        BroadcastPairScalarToVector512,             64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vbroadcasti32x2,    INS_vbroadcasti32x2,    INS_invalid,            INS_invalid,            INS_vbroadcastf32x2,    INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX10v1_V512,        BroadcastVector128ToVector512,              64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vbroadcasti64x2,    INS_vbroadcasti64x2,    INS_invalid,            INS_vbroadcastf64x2},   HW_Category_MemoryLoad,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX10v1_V512,        BroadcastVector256ToVector512,              64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vbroadcasti32x8,    INS_vbroadcasti32x8,    INS_invalid,            INS_invalid,            INS_vbroadcastf32x8,    INS_invalid},           HW_Category_MemoryLoad,             HW_Flag_NoFlag)
HARDWARE_INTRINSIC(AVX10v1_V512,        ConvertToVector256Single,                   64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtqq2ps,          INS_vcvtuqq2ps,         INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        ConvertToVector512Double,                   64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtqq2pd,          INS_vcvtuqq2pd,         INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        ConvertToVector512Int64,                    64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtps2qq,          INS_vcvtpd2qq},         HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        ConvertToVector512Int64WithTruncation,      64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttps2qq,         INS_vcvttpd2qq},        HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        ConvertToVector512UInt64,                   64,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtps2uqq,         INS_vcvtpd2uqq},        HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        ConvertToVector512UInt64WithTruncation,     64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttps2uqq,        INS_vcvttpd2uqq},       HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        DetectConflicts,                            64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vpconflictd,        INS_vpconflictd,        INS_vpconflictq,        INS_vpconflictq,        INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        ExtractVector128,                           64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vextracti64x2,      INS_vextracti64x2,      INS_invalid,            INS_vextractf64x2},     HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        ExtractVector256,                           64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vextracti32x8,      INS_vextracti32x8,      INS_invalid,            INS_invalid,            INS_vextractf32x8,      INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        InsertVector128,                            64,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vinserti64x2,       INS_vinserti64x2,       INS_invalid,            INS_vinsertf64x2},      HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        InsertVector256,                            64,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vinserti32x8,       INS_vinserti32x8,       INS_invalid,            INS_invalid,            INS_vinsertf32x8,       INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        LeadingZeroCount,                           64,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_vplzcntd,           INS_vplzcntd,           INS_vplzcntq,           INS_vplzcntq,           INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        MultiShift,                                 64,              2,     {INS_vpmultishiftqb,    INS_vpmultishiftqb,     INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        MultiplyLow,                                64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vpmullq,            INS_vpmullq,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        Or,                                         64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_orps,               INS_orpd},              HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        PermuteVar64x8,                             64,              2,     {INS_vpermb,            INS_vpermb,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialImport|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        PermuteVar64x8x2,                           64,              3,     {INS_vpermt2b,          INS_vpermt2b,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_PermuteVar2x|HW_Flag_RmwIntrinsic|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        Range,                                      64,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vrangeps,           INS_vrangepd},          HW_Category_IMM,                    HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        Reduce,                                     64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vreduceps,          INS_vreducepd},         HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(AVX10v1_V512,        Xor,                                        64,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_xorps,              INS_xorpd},             HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible|HW_Flag_CanBenefitFromConstantProp)
#define LAST_NI_AVX10v1_V512            NI_AVX10v1_V512_Xor

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  AVX512F.X64 Intrinsics
#define FIRST_NI_AVX10v1_X64        NI_AVX10v1_X64_ConvertScalarToVector128Double
HARDWARE_INTRINSIC(AVX10v1_X64,     ConvertScalarToVector128Double,             16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtsi2sd64,         INS_vcvtusi2sd64,       INS_invalid,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromSecondArg|HW_Flag_CopyUpperBits|HW_Flag_SpecialCodeGen|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX10v1_X64,     ConvertScalarToVector128Single,             16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtsi2ss64,         INS_vcvtusi2ss64,       INS_invalid,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromSecondArg|HW_Flag_CopyUpperBits|HW_Flag_SpecialCodeGen|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX10v1_X64,     ConvertToInt64,                             16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_cvtss2si,           INS_cvtsd2si},          HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX10v1_X64,     ConvertToUInt64,                            16,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvtss2usi,         INS_vcvtsd2usi},        HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen|HW_Flag_EmbRoundingCompatible)
HARDWARE_INTRINSIC(AVX10v1_X64,     ConvertToUInt64WithTruncation,              16,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcvttss2usi64,      INS_vcvttsd2usi64},     HW_Category_SIMDScalar,             HW_Flag_BaseTypeFromFirstArg|HW_Flag_SpecialCodeGen)
#define LAST_NI_AVX10v1_X64         NI_AVX10v1_X64_ConvertToUInt64WithTruncation


// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  AVXVNNI Intrinsics
#define FIRST_NI_AVXVNNI            NI_AVXVNNI_MultiplyWideningAndAdd
HARDWARE_INTRINSIC(AVXVNNI,         MultiplyWideningAndAdd,                     -1,              3,     {INS_invalid,           INS_vpdpbusd,            INS_vpdpwssd,          INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromSecondArg|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVXVNNI,         MultiplyWideningAndAddSaturate,             -1,              3,     {INS_invalid,           INS_vpdpbusds,           INS_vpdpwssds,         INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_BaseTypeFromSecondArg|HW_Flag_NoEvexSemantics)
#define LAST_NI_AVXVNNI             NI_AVXVNNI_MultiplyWideningAndAddSaturate

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  AES Intrinsics
#define FIRST_NI_AES                NI_AES_Decrypt
HARDWARE_INTRINSIC(AES,             Decrypt,                                    16,              2,     {INS_invalid,           INS_aesdec,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AES,             DecryptLast,                                16,              2,     {INS_invalid,           INS_aesdeclast,         INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AES,             Encrypt,                                    16,              2,     {INS_invalid,           INS_aesenc,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AES,             EncryptLast,                                16,              2,     {INS_invalid,           INS_aesenclast,         INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AES,             InverseMixColumns,                          16,              1,     {INS_invalid,           INS_aesimc,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AES,             KeygenAssist,                               16,              2,     {INS_invalid,           INS_aeskeygenassist,    INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_NoEvexSemantics)
#define LAST_NI_AES                 NI_AES_KeygenAssist

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  BMI1 Intrinsics
#define FIRST_NI_BMI1               NI_BMI1_AndNot
HARDWARE_INTRINSIC(BMI1,            AndNot,                                      0,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_andn,               INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_SpecialImport|HW_Flag_NoFloatingPointUsed|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(BMI1,            BitFieldExtract,                             0,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_bextr,              INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_SpecialCodeGen|HW_Flag_SpecialImport|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(BMI1,            ExtractLowestSetBit,                         0,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_blsi,               INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(BMI1,            GetMaskUpToLowestSetBit,                     0,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_blsmsk,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed)
HARDWARE_INTRINSIC(BMI1,            ResetLowestSetBit,                           0,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_blsr,               INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed)
HARDWARE_INTRINSIC(BMI1,            TrailingZeroCount,                           0,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_tzcnt,              INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_SpecialCodeGen)
#define LAST_NI_BMI1                NI_BMI1_TrailingZeroCount

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  BMI1 Intrinsics
#define FIRST_NI_BMI1_X64           NI_BMI1_X64_AndNot
HARDWARE_INTRINSIC(BMI1_X64,        AndNot,                                      0,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_andn,               INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_SpecialImport|HW_Flag_NoFloatingPointUsed|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(BMI1_X64,        BitFieldExtract,                             0,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_bextr,              INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_SpecialCodeGen|HW_Flag_SpecialImport|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(BMI1_X64,        ExtractLowestSetBit,                         0,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_blsi,               INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(BMI1_X64,        GetMaskUpToLowestSetBit,                     0,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_blsmsk,             INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed)
HARDWARE_INTRINSIC(BMI1_X64,        ResetLowestSetBit,                           0,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_blsr,               INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed)
HARDWARE_INTRINSIC(BMI1_X64,        TrailingZeroCount,                           0,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_tzcnt,              INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_SpecialCodeGen)
#define LAST_NI_BMI1_X64            NI_BMI1_X64_TrailingZeroCount

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  BMI2 Intrinsics
#define FIRST_NI_BMI2               NI_BMI2_MultiplyNoFlags
HARDWARE_INTRINSIC(BMI2,            MultiplyNoFlags,                             0,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_mulx,               INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoContainment|HW_Flag_MaybeMemoryStore|HW_Flag_SpecialCodeGen|HW_Flag_NoFloatingPointUsed|HW_Flag_NoEvexSemantics|HW_Flag_MaybeCommutative)
HARDWARE_INTRINSIC(BMI2,            ParallelBitDeposit,                          0,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_pdep,               INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(BMI2,            ParallelBitExtract,                          0,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_pext,               INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(BMI2,            ZeroHighBits,                                0,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_bzhi,               INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_SpecialImport|HW_Flag_NoEvexSemantics)
#define LAST_NI_BMI2                NI_BMI2_ZeroHighBits

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  BMI2 Intrinsics
#define FIRST_NI_BMI2_X64           NI_BMI2_X64_MultiplyNoFlags
HARDWARE_INTRINSIC(BMI2_X64,        MultiplyNoFlags,                             0,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_mulx,               INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoContainment|HW_Flag_MaybeMemoryStore|HW_Flag_SpecialCodeGen|HW_Flag_NoFloatingPointUsed|HW_Flag_NoEvexSemantics|HW_Flag_MaybeCommutative)
HARDWARE_INTRINSIC(BMI2_X64,        ParallelBitDeposit,                          0,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_pdep,               INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(BMI2_X64,        ParallelBitExtract,                          0,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_pext,               INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(BMI2_X64,        ZeroHighBits,                                0,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_bzhi,               INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_SpecialImport|HW_Flag_NoEvexSemantics)
#define LAST_NI_BMI2_X64            NI_BMI2_X64_ZeroHighBits

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  FMA Intrinsics
#define FIRST_NI_FMA                NI_FMA_MultiplyAdd
HARDWARE_INTRINSIC(FMA,             MultiplyAdd,                                -1,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfmadd213ps,        INS_vfmadd213pd},       HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(FMA,             MultiplyAddNegated,                         -1,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfnmadd213ps,       INS_vfnmadd213pd},      HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(FMA,             MultiplyAddNegatedScalar,                   16,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfnmadd213ss,       INS_vfnmadd213sd},      HW_Category_SIMDScalar,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(FMA,             MultiplyAddScalar,                          16,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfmadd213ss,        INS_vfmadd213sd},       HW_Category_SIMDScalar,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(FMA,             MultiplyAddSubtract,                        -1,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfmaddsub213ps,     INS_vfmaddsub213pd},    HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(FMA,             MultiplySubtract,                           -1,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfmsub213ps,        INS_vfmsub213pd},       HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(FMA,             MultiplySubtractAdd,                        -1,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfmsubadd213ps,     INS_vfmsubadd213pd},    HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(FMA,             MultiplySubtractNegated,                    -1,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfnmsub213ps,       INS_vfnmsub213pd},      HW_Category_SimpleSIMD,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_EmbBroadcastCompatible|HW_Flag_EmbMaskingCompatible)
HARDWARE_INTRINSIC(FMA,             MultiplySubtractNegatedScalar,              16,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfnmsub213ss,       INS_vfnmsub213sd},      HW_Category_SIMDScalar,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_CopyUpperBits)
HARDWARE_INTRINSIC(FMA,             MultiplySubtractScalar,                     16,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vfmsub213ss,        INS_vfmsub213sd},       HW_Category_SIMDScalar,             HW_Flag_SpecialCodeGen|HW_Flag_FmaIntrinsic|HW_Flag_RmwIntrinsic|HW_Flag_CopyUpperBits)
#define LAST_NI_FMA                 NI_FMA_MultiplySubtractScalar

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  LZCNT Intrinsics
#define FIRST_NI_LZCNT              NI_LZCNT_LeadingZeroCount
HARDWARE_INTRINSIC(LZCNT,           LeadingZeroCount,                            0,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_lzcnt,              INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_NoRMWSemantics|HW_Flag_SpecialCodeGen)
#define LAST_NI_LZCNT               NI_LZCNT_LeadingZeroCount

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  LZCNT Intrinsics
#define FIRST_NI_LZCNT_X64          NI_LZCNT_X64_LeadingZeroCount
HARDWARE_INTRINSIC(LZCNT_X64,       LeadingZeroCount,                            0,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_lzcnt,              INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_NoRMWSemantics|HW_Flag_SpecialCodeGen)
#define LAST_NI_LZCNT_X64           NI_LZCNT_X64_LeadingZeroCount

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  PCLMULQDQ Intrinsics
#define FIRST_NI_PCLMULQDQ          NI_PCLMULQDQ_CarrylessMultiply
HARDWARE_INTRINSIC(PCLMULQDQ,       CarrylessMultiply,                          16,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_pclmulqdq,          INS_pclmulqdq,          INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_NoEvexSemantics)
#define LAST_NI_PCLMULQDQ           NI_PCLMULQDQ_CarrylessMultiply

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  POPCNT Intrinsics
#define FIRST_NI_POPCNT             NI_POPCNT_PopCount
HARDWARE_INTRINSIC(POPCNT,          PopCount,                                    0,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_popcnt,             INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_NoRMWSemantics|HW_Flag_SpecialCodeGen)
#define LAST_NI_POPCNT              NI_POPCNT_PopCount

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  POPCNT Intrinsics
#define FIRST_NI_POPCNT_X64         NI_POPCNT_X64_PopCount
HARDWARE_INTRINSIC(POPCNT_X64,      PopCount,                                    0,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_popcnt,             INS_invalid,            INS_invalid},           HW_Category_Scalar,                 HW_Flag_NoFloatingPointUsed|HW_Flag_NoRMWSemantics|HW_Flag_SpecialCodeGen)
#define LAST_NI_POPCNT_X64          NI_POPCNT_X64_PopCount

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//  X86Serialize Intrinsics
#define FIRST_NI_X86Serialize       NI_X86Serialize_Serialize
HARDWARE_INTRINSIC(X86Serialize,    Serialize,                                   0,              0,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment|HW_Flag_NoRMWSemantics|HW_Flag_SpecialSideEffect_Barrier)
#define LAST_NI_X86Serialize        NI_X86Serialize_Serialize

// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
//                 ISA              Function name                               SIMD size       NumArg                                                                                                                            Instructions                                                                                                                  Category                            Flags
//                                                                                                      {TYP_BYTE,              TYP_UBYTE,              TYP_SHORT,              TYP_USHORT,             TYP_INT,                TYP_UINT,               TYP_LONG,               TYP_ULONG,              TYP_FLOAT,              TYP_DOUBLE}
// ***************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************************
// Special intrinsics that are generated during lowering
HARDWARE_INTRINSIC(SSE,             COMISS,                                     16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_comiss,             INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE,             UCOMISS,                                    16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_ucomiss,            INS_invalid},           HW_Category_SIMDScalar,             HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            COMISD,                                     16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_comisd},            HW_Category_SIMDScalar,             HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE2,            UCOMISD,                                    16,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_ucomisd},           HW_Category_SIMDScalar,             HW_Flag_NoRMWSemantics)
HARDWARE_INTRINSIC(SSE41,           PTEST,                                      16,              2,     {INS_ptest,             INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_invalid,            INS_invalid},           HW_Category_SimpleSIMD,             HW_Flag_NoRMWSemantics|HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(AVX,             PTEST,                                       0,              2,     {INS_ptest,             INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_ptest,              INS_vtestps,            INS_vtestpd},           HW_Category_SimpleSIMD,             HW_Flag_NoEvexSemantics)
HARDWARE_INTRINSIC(EVEX,            KORTEST,                                     0,             -1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment)
HARDWARE_INTRINSIC(EVEX,            KTEST,                                       0,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment)
HARDWARE_INTRINSIC(EVEX,            PTESTM,                                      0,              2,     {INS_vptestmb,          INS_vptestmb,           INS_vptestmw,           INS_vptestmw,           INS_vptestmd,           INS_vptestmd,           INS_vptestmq,           INS_vptestmq,           INS_vptestmd,           INS_vptestmq},          HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible)
HARDWARE_INTRINSIC(EVEX,            PTESTNM,                                     0,              2,     {INS_vptestnmb,         INS_vptestnmb,          INS_vptestnmw,          INS_vptestnmw,          INS_vptestnmd,          INS_vptestnmd,          INS_vptestnmq,          INS_vptestnmq,          INS_vptestnmd,          INS_vptestnmq},         HW_Category_SimpleSIMD,             HW_Flag_Commutative|HW_Flag_EmbBroadcastCompatible)

HARDWARE_INTRINSIC(EVEX,            AddMask,                                    -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment|HW_Flag_Commutative|HW_Flag_ReturnsPerElementMask)
HARDWARE_INTRINSIC(EVEX,            AndMask,                                    -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment|HW_Flag_Commutative|HW_Flag_ReturnsPerElementMask)
HARDWARE_INTRINSIC(EVEX,            AndNotMask,                                 -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment|HW_Flag_ReturnsPerElementMask)
HARDWARE_INTRINSIC(EVEX,            BlendVariableMask,                          -1,              3,     {INS_vpblendmb,         INS_vpblendmb,          INS_vpblendmw,          INS_vpblendmw,          INS_vpblendmd,          INS_vpblendmd,          INS_vpblendmq,          INS_vpblendmq,          INS_vblendmps,          INS_vblendmpd},         HW_Category_SimpleSIMD,             HW_Flag_EmbBroadcastCompatible)
HARDWARE_INTRINSIC(EVEX,            CompareMask,                                -1,              3,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcmpps,             INS_vcmppd},            HW_Category_IMM,                    HW_Flag_ReturnsPerElementMask|HW_Flag_EmbBroadcastCompatible)
HARDWARE_INTRINSIC(EVEX,            CompareEqualMask,                           -1,              2,     {INS_vpcmpeqb,          INS_vpcmpeqb,           INS_vpcmpeqw,           INS_vpcmpeqw,           INS_vpcmpeqd,           INS_vpcmpeqd,           INS_vpcmpeqq,           INS_vpcmpeqq,           INS_vcmpps,             INS_vcmppd},            HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_EmbBroadcastCompatible|HW_Flag_Commutative|HW_Flag_CanBenefitFromConstantProp)
HARDWARE_INTRINSIC(EVEX,            CompareGreaterThanMask,                     -1,              2,     {INS_vpcmpgtb,          INS_vpcmpub,            INS_vpcmpgtw,           INS_vpcmpuw,            INS_vpcmpgtd,           INS_vpcmpud,            INS_vpcmpgtq,           INS_vpcmpuq,            INS_vcmpps,             INS_vcmppd},            HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_EmbBroadcastCompatible)
HARDWARE_INTRINSIC(EVEX,            CompareGreaterThanOrEqualMask,              -1,              2,     {INS_vpcmpb,            INS_vpcmpub,            INS_vpcmpw,             INS_vpcmpuw,            INS_vpcmpd,             INS_vpcmpud,            INS_vpcmpq,             INS_vpcmpuq,            INS_vcmpps,             INS_vcmppd},            HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_EmbBroadcastCompatible)
HARDWARE_INTRINSIC(EVEX,            CompareLessThanMask,                        -1,              2,     {INS_vpcmpb,            INS_vpcmpub,            INS_vpcmpw,             INS_vpcmpuw,            INS_vpcmpd,             INS_vpcmpud,            INS_vpcmpq,             INS_vpcmpuq,            INS_vcmpps,             INS_vcmppd},            HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_EmbBroadcastCompatible)
HARDWARE_INTRINSIC(EVEX,            CompareLessThanOrEqualMask,                 -1,              2,     {INS_vpcmpb,            INS_vpcmpub,            INS_vpcmpw,             INS_vpcmpuw,            INS_vpcmpd,             INS_vpcmpud,            INS_vpcmpq,             INS_vpcmpuq,            INS_vcmpps,             INS_vcmppd},            HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_EmbBroadcastCompatible)
HARDWARE_INTRINSIC(EVEX,            CompareNotEqualMask,                        -1,              2,     {INS_vpcmpb,            INS_vpcmpub,            INS_vpcmpw,             INS_vpcmpuw,            INS_vpcmpd,             INS_vpcmpud,            INS_vpcmpq,             INS_vpcmpuq,            INS_vcmpps,             INS_vcmppd},            HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_EmbBroadcastCompatible|HW_Flag_Commutative|HW_Flag_CanBenefitFromConstantProp)
HARDWARE_INTRINSIC(EVEX,            CompareNotGreaterThanMask,                  -1,              2,     {INS_vpcmpb,            INS_vpcmpub,            INS_vpcmpw,             INS_vpcmpuw,            INS_vpcmpd,             INS_vpcmpud,            INS_vpcmpq,             INS_vpcmpuq,            INS_vcmpps,             INS_vcmppd},            HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_EmbBroadcastCompatible)
HARDWARE_INTRINSIC(EVEX,            CompareNotGreaterThanOrEqualMask,           -1,              2,     {INS_vpcmpb,            INS_vpcmpub,            INS_vpcmpw,             INS_vpcmpuw,            INS_vpcmpd,             INS_vpcmpud,            INS_vpcmpq,             INS_vpcmpuq,            INS_vcmpps,             INS_vcmppd},            HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_EmbBroadcastCompatible)
HARDWARE_INTRINSIC(EVEX,            CompareNotLessThanMask,                     -1,              2,     {INS_vpcmpb,            INS_vpcmpub,            INS_vpcmpw,             INS_vpcmpuw,            INS_vpcmpd,             INS_vpcmpud,            INS_vpcmpq,             INS_vpcmpuq,            INS_vcmpps,             INS_vcmppd},            HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_EmbBroadcastCompatible)
HARDWARE_INTRINSIC(EVEX,            CompareNotLessThanOrEqualMask,              -1,              2,     {INS_vpcmpb,            INS_vpcmpub,            INS_vpcmpw,             INS_vpcmpuw,            INS_vpcmpd,             INS_vpcmpud,            INS_vpcmpq,             INS_vpcmpuq,            INS_vcmpps,             INS_vcmppd},            HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_EmbBroadcastCompatible)
HARDWARE_INTRINSIC(EVEX,            CompareOrderedMask,                         -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcmpps,             INS_vcmppd},            HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_EmbBroadcastCompatible)
HARDWARE_INTRINSIC(EVEX,            CompareUnorderedMask,                       -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_vcmpps,             INS_vcmppd},            HW_Category_SimpleSIMD,             HW_Flag_ReturnsPerElementMask|HW_Flag_EmbBroadcastCompatible)
HARDWARE_INTRINSIC(EVEX,            ConvertMaskToVector,                        -1,              1,     {INS_vpmovm2b,          INS_vpmovm2b,           INS_vpmovm2w,           INS_vpmovm2w,           INS_vpmovm2d,           INS_vpmovm2d,           INS_vpmovm2q,           INS_vpmovm2q,           INS_vpmovm2d,           INS_vpmovm2q},          HW_Category_SimpleSIMD,             HW_Flag_NoContainment|HW_Flag_ReturnsPerElementMask)
HARDWARE_INTRINSIC(EVEX,            ConvertVectorToMask,                        -1,              1,     {INS_vpmovb2m,          INS_vpmovb2m,           INS_vpmovw2m,           INS_vpmovw2m,           INS_vpmovd2m,           INS_vpmovd2m,           INS_vpmovq2m,           INS_vpmovq2m,           INS_vpmovd2m,           INS_vpmovq2m},          HW_Category_SimpleSIMD,             HW_Flag_NoContainment|HW_Flag_ReturnsPerElementMask)
HARDWARE_INTRINSIC(EVEX,            MoveMask,                                   -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment)
HARDWARE_INTRINSIC(EVEX,            NotMask,                                    -1,              1,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment|HW_Flag_ReturnsPerElementMask)
HARDWARE_INTRINSIC(EVEX,            op_EqualityMask,                            -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment|HW_Flag_Commutative)
HARDWARE_INTRINSIC(EVEX,            op_InequalityMask,                          -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment|HW_Flag_Commutative)
HARDWARE_INTRINSIC(EVEX,            OrMask,                                     -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment|HW_Flag_Commutative|HW_Flag_ReturnsPerElementMask)
HARDWARE_INTRINSIC(EVEX,            ShiftLeftMask,                              -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(EVEX,            ShiftRightMask,                             -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_IMM,                    HW_Flag_FullRangeIMM|HW_Flag_SpecialCodeGen)
HARDWARE_INTRINSIC(EVEX,            XorMask,                                    -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment|HW_Flag_Commutative|HW_Flag_ReturnsPerElementMask)
HARDWARE_INTRINSIC(EVEX,            XnorMask,                                   -1,              2,     {INS_invalid,           INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid,            INS_invalid},           HW_Category_Special,                HW_Flag_NoContainment|HW_Flag_Commutative|HW_Flag_ReturnsPerElementMask)

#endif // FEATURE_HW_INTRINSIC

#undef HARDWARE_INTRINSIC

// clang-format on
