// Copyright 2017 The Chromium Authors. All rights reserved.
// Copyright (C) 2018 Apple Inc. All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//    * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//    * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//    * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

#include "config.h"
#include "ApduResponse.h"

#include <wtf/Optional.h>

#if ENABLE(WEB_AUTHN)

namespace apdu {

// static
Optional<ApduResponse> ApduResponse::createFromMessage(const Vector<uint8_t>& data)
{
    // Invalid message size, data is appended by status byte.
    if (data.size() < 2)
        return WTF::nullopt;

    uint16_t statusBytes = data[data.size() - 2] << 8;
    statusBytes |= data[data.size() - 1];

    Vector<uint8_t> newData;
    newData.appendRange(data.begin(), data.end() - 2);
    return ApduResponse(WTFMove(newData), static_cast<Status>(statusBytes));
}

ApduResponse::ApduResponse(Vector<uint8_t>&& data, Status responseStatus)
    : m_data(WTFMove(data))
    , m_responseStatus(responseStatus)
{
}

Vector<uint8_t> ApduResponse::getEncodedResponse() const
{
    Vector<uint8_t> encodedResponse = m_data;
    encodedResponse.append(static_cast<uint16_t>(m_responseStatus) >> 8 & 0xff);
    encodedResponse.append(static_cast<uint16_t>(m_responseStatus) & 0xff);
    return encodedResponse;
}

} // namespace apdu

#endif // ENABLE(WEB_AUTHN)
